/*
 * ESP32 Peripheral Manager Usage Examples
 * 
 * This file demonstrates common usage patterns for the ESP32 Peripheral Manager API
 */

#include "esp32-hal-periman.h"
#include "esp32-hal-log.h"
#include "driver/i2c.h"
#include "driver/gpio.h"

// Example 1: I2C Peripheral Setup with Error Handling
static bool i2c_deinit_callback(void *bus) {
    i2c_port_t port = (i2c_port_t)((int)bus - 1);
    esp_err_t err = i2c_driver_delete(port);
    if (err != ESP_OK) {
        log_e("I2C driver delete failed: %s", esp_err_to_name(err));
        return false;
    }
    log_i("I2C port %d successfully deinitialized", port);
    return true;
}

bool setup_i2c_example(uint8_t sda_pin, uint8_t scl_pin, i2c_port_t port) {
    log_i("Setting up I2C on SDA=%d, SCL=%d, Port=%d", sda_pin, scl_pin, port);
    
    // Step 1: Register deinit callbacks
    perimanSetBusDeinit(ESP32_BUS_TYPE_I2C_MASTER_SDA, i2c_deinit_callback);
    perimanSetBusDeinit(ESP32_BUS_TYPE_I2C_MASTER_SCL, i2c_deinit_callback);
    
    // Step 2: Clear any existing assignments
    if (!perimanClearPinBus(sda_pin) || !perimanClearPinBus(scl_pin)) {
        log_e("Failed to clear I2C pins");
        return false;
    }
    
    // Step 3: Configure I2C hardware
    i2c_config_t conf = {
        .mode = I2C_MODE_MASTER,
        .sda_io_num = sda_pin,
        .scl_io_num = scl_pin,
        .sda_pullup_en = GPIO_PULLUP_ENABLE,
        .scl_pullup_en = GPIO_PULLUP_ENABLE,
        .master.clk_speed = 100000
    };
    
    esp_err_t err = i2c_param_config(port, &conf);
    if (err != ESP_OK) {
        log_e("I2C param config failed: %s", esp_err_to_name(err));
        return false;
    }
    
    err = i2c_driver_install(port, conf.mode, 0, 0, 0);
    if (err != ESP_OK) {
        log_e("I2C driver install failed: %s", esp_err_to_name(err));
        return false;
    }
    
    // Step 4: Register pins with peripheral manager
    void* bus_handle = (void*)(port + 1);
    if (!perimanSetPinBus(sda_pin, ESP32_BUS_TYPE_I2C_MASTER_SDA, 
                          bus_handle, port, -1) ||
        !perimanSetPinBus(scl_pin, ESP32_BUS_TYPE_I2C_MASTER_SCL, 
                          bus_handle, port, -1)) {
        log_e("Failed to register I2C pins with peripheral manager");
        i2c_driver_delete(port);
        return false;
    }
    
    log_i("I2C setup complete on pins SDA=%d, SCL=%d", sda_pin, scl_pin);
    return true;
}

// Example 2: Pin Conflict Detection and Resolution
bool check_pin_conflicts(uint8_t* pins, size_t count, peripheral_bus_type_t intended_type) {
    bool conflicts_found = false;
    
    log_i("Checking pin conflicts for %zu pins", count);
    
    for (size_t i = 0; i < count; i++) {
        peripheral_bus_type_t current = perimanGetPinBusType(pins[i]);
        
        if (current == ESP32_BUS_TYPE_INIT) {
            log_v("Pin %d is available", pins[i]);
            continue;
        }
        
        if (current == intended_type) {
            log_w("Pin %d already assigned to %s (same type)", 
                  pins[i], perimanGetTypeName(current));
            continue;
        }
        
        // Conflict detected
        log_e("Pin %d conflict: currently %s, requested %s",
              pins[i], 
              perimanGetTypeName(current),
              perimanGetTypeName(intended_type));
        conflicts_found = true;
    }
    
    return !conflicts_found;
}

bool resolve_pin_conflicts(uint8_t* pins, size_t count) {
    log_i("Resolving conflicts for %zu pins", count);
    
    for (size_t i = 0; i < count; i++) {
        peripheral_bus_type_t current = perimanGetPinBusType(pins[i]);
        
        if (current != ESP32_BUS_TYPE_INIT) {
            log_i("Clearing pin %d from %s", pins[i], perimanGetTypeName(current));
            if (!perimanClearPinBus(pins[i])) {
                log_e("Failed to clear pin %d", pins[i]);
                return false;
            }
        }
    }
    
    log_i("All pin conflicts resolved");
    return true;
}

// Example 3: Safe Pin Switching Between Peripherals
bool switch_to_gpio(uint8_t pin, gpio_mode_t mode) {
    peripheral_bus_type_t current = perimanGetPinBusType(pin);
    
    if (current == ESP32_BUS_TYPE_GPIO) {
        log_i("Pin %d already configured as GPIO", pin);
        return true;
    }
    
    if (current != ESP32_BUS_TYPE_INIT) {
        log_i("Switching pin %d from %s to GPIO", pin, perimanGetTypeName(current));
        if (!perimanClearPinBus(pin)) {
            log_e("Failed to clear pin %d", pin);
            return false;
        }
    }
    
    // Configure GPIO hardware
    gpio_config_t gpio_conf = {
        .pin_bit_mask = (1ULL << pin),
        .mode = mode,
        .pull_up_en = GPIO_PULLUP_DISABLE,
        .pull_down_en = GPIO_PULLDOWN_DISABLE,
        .intr_type = GPIO_INTR_DISABLE
    };
    
    esp_err_t err = gpio_config(&gpio_conf);
    if (err != ESP_OK) {
        log_e("GPIO configuration failed for pin %d: %s", pin, esp_err_to_name(err));
        return false;
    }
    
    // Register with peripheral manager
    if (!perimanSetPinBus(pin, ESP32_BUS_TYPE_GPIO, NULL, -1, -1)) {
        log_e("Failed to register pin %d as GPIO", pin);
        return false;
    }
    
    log_i("Pin %d successfully configured as GPIO", pin);
    return true;
}

// Example 4: Multi-peripheral Sensor Setup
typedef struct {
    uint8_t sda_pin;
    uint8_t scl_pin;
    uint8_t int_pin;
    uint8_t rst_pin;
    i2c_port_t i2c_port;
    bool initialized;
} sensor_config_t;

static bool gpio_deinit_callback(void *bus) {
    // GPIO pins don't need special cleanup in most cases
    log_v("GPIO pin deinitializing");
    return true;
}

bool init_sensor_with_extras(sensor_config_t* config) {
    if (config == NULL) {
        log_e("Invalid config pointer");
        return false;
    }
    
    log_i("Initializing sensor with I2C + GPIO pins");
    
    // Register all deinit callbacks
    perimanSetBusDeinit(ESP32_BUS_TYPE_I2C_MASTER_SDA, i2c_deinit_callback);
    perimanSetBusDeinit(ESP32_BUS_TYPE_I2C_MASTER_SCL, i2c_deinit_callback);
    perimanSetBusDeinit(ESP32_BUS_TYPE_GPIO, gpio_deinit_callback);
    
    // Create pin array for conflict checking
    uint8_t pins[] = {config->sda_pin, config->scl_pin, config->int_pin, config->rst_pin};
    size_t pin_count = sizeof(pins) / sizeof(pins[0]);
    
    // Check for conflicts and resolve if necessary
    if (!resolve_pin_conflicts(pins, pin_count)) {
        log_e("Failed to resolve pin conflicts");
        return false;
    }
    
    // Initialize I2C
    if (!setup_i2c_example(config->sda_pin, config->scl_pin, config->i2c_port)) {
        log_e("I2C initialization failed");
        return false;
    }
    
    // Configure interrupt pin as input with pullup
    gpio_config_t int_conf = {
        .pin_bit_mask = (1ULL << config->int_pin),
        .mode = GPIO_MODE_INPUT,
        .pull_up_en = GPIO_PULLUP_ENABLE,
        .pull_down_en = GPIO_PULLDOWN_DISABLE,
        .intr_type = GPIO_INTR_NEGEDGE
    };
    
    if (gpio_config(&int_conf) != ESP_OK) {
        log_e("Interrupt GPIO configuration failed");
        goto cleanup_i2c;
    }
    
    // Configure reset pin as output
    gpio_config_t rst_conf = {
        .pin_bit_mask = (1ULL << config->rst_pin),
        .mode = GPIO_MODE_OUTPUT,
        .pull_up_en = GPIO_PULLUP_DISABLE,
        .pull_down_en = GPIO_PULLDOWN_DISABLE,
        .intr_type = GPIO_INTR_DISABLE
    };
    
    if (gpio_config(&rst_conf) != ESP_OK) {
        log_e("Reset GPIO configuration failed");
        goto cleanup_i2c;
    }
    
    // Register GPIO pins with peripheral manager
    if (!perimanSetPinBus(config->int_pin, ESP32_BUS_TYPE_GPIO, NULL, -1, -1) ||
        !perimanSetPinBus(config->rst_pin, ESP32_BUS_TYPE_GPIO, NULL, -1, -1)) {
        log_e("Failed to register GPIO pins");
        goto cleanup_i2c;
    }
    
    // Perform sensor reset sequence
    gpio_set_level(config->rst_pin, 0);
    vTaskDelay(pdMS_TO_TICKS(10));
    gpio_set_level(config->rst_pin, 1);
    vTaskDelay(pdMS_TO_TICKS(50));
    
    config->initialized = true;
    log_i("Sensor initialized successfully");
    return true;
    
cleanup_i2c:
    // Cleanup will be handled automatically by peripheral manager callbacks
    perimanClearPinBus(config->sda_pin);
    perimanClearPinBus(config->scl_pin);
    return false;
}

void cleanup_sensor(sensor_config_t* config) {
    if (config == NULL || !config->initialized) {
        return;
    }
    
    log_i("Cleaning up sensor configuration");
    
    // Clear all pins - this will trigger the appropriate deinit callbacks
    perimanClearPinBus(config->sda_pin);
    perimanClearPinBus(config->scl_pin);
    perimanClearPinBus(config->int_pin);
    perimanClearPinBus(config->rst_pin);
    
    config->initialized = false;
    log_i("Sensor cleanup complete");
}

// Example 5: Debugging and Monitoring Functions
void debug_all_pin_assignments(void) {
    log_i("=== Current GPIO Pin Assignments ===");
    
    int assigned_count = 0;
    for (int pin = 0; pin < SOC_GPIO_PIN_COUNT; pin++) {
        peripheral_bus_type_t type = perimanGetPinBusType(pin);
        
        if (type != ESP32_BUS_TYPE_INIT) {
            void* bus = perimanGetPinBus(pin, type);
            log_i("Pin %2d: %-20s (bus: %p)", 
                  pin, perimanGetTypeName(type), bus);
            assigned_count++;
        }
    }
    
    log_i("=== Total assigned pins: %d ===", assigned_count);
}

void debug_specific_peripheral(peripheral_bus_type_t type) {
    log_i("=== Pins assigned to %s ===", perimanGetTypeName(type));
    
    int count = 0;
    for (int pin = 0; pin < SOC_GPIO_PIN_COUNT; pin++) {
        if (perimanGetPinBusType(pin) == type) {
            void* bus = perimanGetPinBus(pin, type);
            log_i("Pin %2d: bus handle %p", pin, bus);
            count++;
        }
    }
    
    if (count == 0) {
        log_i("No pins assigned to %s", perimanGetTypeName(type));
    } else {
        log_i("Total pins: %d", count);
    }
}

bool validate_pin_assignment(uint8_t pin, peripheral_bus_type_t expected_type) {
    peripheral_bus_type_t actual_type = perimanGetPinBusType(pin);
    
    if (actual_type != expected_type) {
        log_e("Pin %d validation failed: expected %s, got %s",
              pin, perimanGetTypeName(expected_type), perimanGetTypeName(actual_type));
        return false;
    }
    
    void* bus = perimanGetPinBus(pin, expected_type);
    if (bus == NULL && expected_type != ESP32_BUS_TYPE_INIT) {
        log_w("Pin %d has correct type but NULL bus handle", pin);
    }
    
    log_v("Pin %d validation passed: %s", pin, perimanGetTypeName(expected_type));
    return true;
}

// Example 6: Arduino-style Usage Pattern
/*
// This would typically be in a .cpp file for Arduino

class MyPeripheral {
private:
    uint8_t pin;
    peripheral_bus_type_t bus_type;
    void* bus_handle;
    bool initialized;
    
    static bool deinit_callback(void* bus) {
        log_i("MyPeripheral deinit callback called");
        return true;
    }
    
public:
    MyPeripheral() : pin(255), bus_type(ESP32_BUS_TYPE_INIT), 
                     bus_handle(nullptr), initialized(false) {}
    
    bool begin(uint8_t pin_num, peripheral_bus_type_t type = ESP32_BUS_TYPE_GPIO) {
        if (initialized) {
            log_w("MyPeripheral already initialized");
            return true;
        }
        
        // Register deinit callback
        perimanSetBusDeinit(type, deinit_callback);
        
        // Check pin availability
        if (perimanGetPinBusType(pin_num) != ESP32_BUS_TYPE_INIT) {
            log_i("Pin %d in use, clearing...", pin_num);
            if (!perimanClearPinBus(pin_num)) {
                log_e("Failed to clear pin %d", pin_num);
                return false;
            }
        }
        
        // Initialize peripheral hardware here
        // ...
        
        // Create bus handle
        bus_handle = (void*)(pin_num + 1); // Simple example handle
        
        // Register with peripheral manager
        if (!perimanSetPinBus(pin_num, type, bus_handle, -1, -1)) {
            log_e("Failed to register pin %d", pin_num);
            return false;
        }
        
        pin = pin_num;
        bus_type = type;
        initialized = true;
        
        log_i("MyPeripheral initialized on pin %d", pin);
        return true;
    }
    
    void end() {
        if (initialized && pin != 255) {
            perimanClearPinBus(pin);
            pin = 255;
            bus_type = ESP32_BUS_TYPE_INIT;
            bus_handle = nullptr;
            initialized = false;
            log_i("MyPeripheral deinitialized");
        }
    }
    
    bool isInitialized() const { return initialized; }
    uint8_t getPin() const { return pin; }
    peripheral_bus_type_t getBusType() const { return bus_type; }
};
*/

// Example usage function
void example_usage_demo() {
    log_i("Starting ESP32 Peripheral Manager examples");
    
    // Example 1: Basic I2C setup
    if (setup_i2c_example(21, 22, I2C_NUM_0)) {
        log_i("✓ I2C example completed successfully");
    } else {
        log_e("✗ I2C example failed");
    }
    
    // Example 2: Pin conflict checking
    uint8_t test_pins[] = {21, 22, 23, 24};
    if (check_pin_conflicts(test_pins, 4, ESP32_BUS_TYPE_GPIO)) {
        log_i("✓ Pin conflict check passed");
    } else {
        log_w("⚠ Pin conflicts detected");
    }
    
    // Example 3: Debug current assignments
    debug_all_pin_assignments();
    
    // Example 4: Sensor setup
    sensor_config_t sensor = {
        .sda_pin = 21,
        .scl_pin = 22,
        .int_pin = 23,
        .rst_pin = 24,
        .i2c_port = I2C_NUM_0,
        .initialized = false
    };
    
    if (init_sensor_with_extras(&sensor)) {
        log_i("✓ Sensor example completed successfully");
        
        // Clean up
        cleanup_sensor(&sensor);
    } else {
        log_e("✗ Sensor example failed");
    }
    
    log_i("ESP32 Peripheral Manager examples completed");
}