pub mod services;

use ::time::format_description;
use miette::{IntoDiagnostic, Result};
use rmcp::{ServiceExt, service::RunningService, transport};
use services::McpService;
use tokio_graceful_shutdown::SubsystemHandle;
use tracing::{Level, info};
use tracing_appender::{
    non_blocking::WorkerGuard,
    rolling::{RollingFileAppender, Rotation},
};
use tracing_subscriber::{
    fmt::{self, time::UtcTime, writer::MakeWriterExt},
    layer::SubscriberExt,
    util::SubscriberInitExt,
};

const NAME: &str = env!("CARGO_PKG_NAME");
const LOG_DIR: &str = "logs";

pub fn init_logger() -> Result<WorkerGuard> {
    let time_format = format_description::parse_borrowed::<2>(
        "[year]-[month]-[day]T[hour]:[minute]:[second].[subsecond digits:3]Z",
    )
    .expect("format string should be valid!");
    let timer = UtcTime::new(time_format);

    let file_appender = RollingFileAppender::builder()
        .rotation(Rotation::DAILY)
        .filename_prefix(NAME)
        .filename_suffix("log")
        .max_log_files(3)
        .build(LOG_DIR)
        .expect("failed to create log file appender");

    let (non_blocking, guard) = tracing_appender::non_blocking(file_appender);

    let stderr_layer = fmt::layer()
        .with_ansi(false)
        .with_timer(timer.clone())
        .with_writer(std::io::stderr.with_max_level(Level::DEBUG));

    let file_layer = fmt::layer()
        .with_ansi(false)
        .with_timer(timer)
        .with_writer(non_blocking.with_max_level(Level::DEBUG));

    let tokio_console_layer = console_subscriber::spawn();

    tracing_subscriber::registry()
        .with(stderr_layer)
        .with(file_layer)
        .with(tokio_console_layer)
        .init();

    info!("Tracing initialized successfully");
    Ok(guard)
}

pub async fn start_mcp_server(
    subsys: SubsystemHandle<Box<dyn std::error::Error + Send + Sync>>,
) -> Result<()> {
    let service = McpService::new();
    info!("Starting MCP service");
    let cancelation_token = subsys.create_cancellation_token();
    let server: RunningService<_, _> = service
        .serve_with_ct(transport::stdio(), cancelation_token)
        .await
        .into_diagnostic()?;
    info!("MCP server initialized");
    let quit_reason = server.waiting().await.into_diagnostic()?;
    info!("MCP server shutdown with reason: {:?}", quit_reason);
    Ok(())
}
