package go_sqlite_bench

import (
	"database/sql"
	"fmt"
	"path"
	"strings"
	"testing"
)

func BenchmarkDB_Subquery(b *testing.B) {
	testCases := []struct {
		posts             int
		postParagraphs    int
		commentsPerPost   int
		commentParagraphs int
	}{
		{100, 1, 25, 1},
		{100, 1, 25, 100},
		{100, 100, 25, 1},
		{100, 100, 25, 100},
	}

	for _, tc := range testCases {
		b.Run(fmt.Sprintf("posts=%d,postParagraphs=%d,commentsPerPost=%d,commentParagraphs=%d", tc.posts, tc.postParagraphs, tc.commentsPerPost, tc.commentParagraphs), func(b *testing.B) {
			db, err := OpenDB(path.Join(b.TempDir(), "go-sqlite-bench.db"))
			if err != nil {
				noErr(b, err)
			}
			defer db.Close()

			err = prepareDB(db)
			if err != nil {
				noErr(b, err)
			}

			err = populateDB(db, tc.posts, tc.postParagraphs, tc.commentsPerPost, tc.commentParagraphs)
			if err != nil {
				noErr(b, err)
			}

			b.ResetTimer()

			b.RunParallel(func(pb *testing.PB) {
				for pb.Next() {
					_, err := db.Exec(`
									SELECT
											id,
											title,
											(SELECT COUNT(*) FROM comments WHERE post_id = posts.id) as comment_count
											-- ,
											-- (SELECT AVG(LENGTH(content)) FROM comments WHERE post_id = posts.id) AS avg_comment_length
											-- ,
											-- (SELECT MAX(LENGTH(content)) FROM comments WHERE post_id = posts.id) AS max_comment_length
									FROM posts;
									`)
					noErr(b, err)
				}
			})
		})
	}
}

func prepareDB(db *sql.DB) error {
	_, err := db.Exec(`
	create table posts (
		id integer primary key,
		title text not null,
		content text not null
	);
	`)
	if err != nil {
		return err
	}

	_, err = db.Exec(`
	create table comments (
		id integer primary key,
		post_id int not null references posts (id),
		name text not null,
		content text not null
	);
	`)
	if err != nil {
		return err
	}

	return nil
}

func populateDB(db *sql.DB, posts, postParagraphs, commentsPerPost, commentParagraphs int) error {
	loremIpsum := `Lorem ipsum dolor sit amet, consectetur adipiscing elit. Duis eget sapien accumsan, commodo ligula iaculis, pretium ligula. In ac lobortis nulla. Donec lobortis metus sed mauris iaculis euismod. Ut vehicula velit vitae dolor maximus euismod. Nulla vel risus eros. Vivamus porttitor odio eleifend, imperdiet tellus sed, feugiat augue. Donec faucibus eget nunc facilisis gravida. Fusce posuere ac lacus eu rutrum. Vivamus nec nibh sed nisl maximus varius. Pellentesque in placerat eros. Vivamus efficitur in dolor nec eleifend. Proin quis nibh quis enim rutrum posuere. Aliquam odio metus, scelerisque quis massa imperdiet, tincidunt placerat orci. Maecenas posuere, ex vitae porttitor tristique, risus erat bibendum augue, nec tempus eros dolor vitae ipsum.`
	postContent := paragraphs(loremIpsum, postParagraphs)
	commentContent := paragraphs(loremIpsum, commentParagraphs)

	for i := range posts {
		title := fmt.Sprintf("Post %d", i)
		content := postContent

		_, err := db.Exec(`insert into posts (id, title, content) values ($1, $2, $3)`, i, title, content)
		if err != nil {
			return err
		}

		for j := range commentsPerPost {
			name := fmt.Sprintf("Name %d.%d", i, j)
			content := commentContent

			_, err := db.Exec(`insert into comments (post_id, name, content) values ($1, $2, $3)`, i, name, content)
			if err != nil {
				return err
			}
		}
	}

	return nil
}

func paragraphs(s string, n int) string {
	paragraphs := make([]string, n)

	for i := range n {
		paragraphs[i] = s
	}

	return strings.Join(paragraphs, "\n\n")
}

func noErr(tb testing.TB, err error) {
	tb.Helper()

	if err != nil {
		tb.Fatal("Error is not nil:", err)
	}
}
