package com.example.demo

import org.springframework.boot.autoconfigure.SpringBootApplication
import org.springframework.boot.runApplication
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Configuration
import org.springframework.http.HttpStatus
import org.springframework.security.config.annotation.web.builders.HttpSecurity
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity
import org.springframework.security.config.web.servlet.invoke
import org.springframework.security.web.DefaultRedirectStrategy
import org.springframework.security.web.SecurityFilterChain
import org.springframework.security.web.savedrequest.CookieRequestCache
import org.springframework.security.web.savedrequest.RequestCache
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.RestController
import javax.servlet.http.HttpServletRequest
import javax.servlet.http.HttpServletResponse

@Configuration(proxyBeanMethods = false)
class RequestCacheConfiguration {
    @Bean
    fun requestCache(): RequestCache = CookieRequestCache()
}

@EnableWebSecurity
class SecurityConfiguration {
    @Bean
    fun securityFilterChain(http: HttpSecurity, requestCache: RequestCache): SecurityFilterChain {
        http {
            authorizeRequests { authorize(anyRequest, permitAll) }
            requestCache { this.requestCache = requestCache }
        }
        return http.build()
    }
}


@RestController
class Controller(private val requestCache: RequestCache) {
    private val redirectStrategy = DefaultRedirectStrategy()

    @GetMapping("/")
    fun index(
        @RequestParam goto: String,
        request: HttpServletRequest,
        response: HttpServletResponse
    ) {
        if (request.session.getAttribute("redirected") != "true") {
            request.session.setAttribute("redirected", "true")
            requestCache.saveRequest(request, response)
            redirectStrategy.sendRedirect(request, response, "/redirect")
        } else {
            request.session.removeAttribute("redirected")
            response.status = HttpStatus.OK.value()
            response.writer.println(goto)
        }
    }

    @GetMapping("/redirect")
    fun redirect(request: HttpServletRequest, response: HttpServletResponse) {
        val savedRequest = requestCache.getRequest(request, response)
        redirectStrategy.sendRedirect(request, response, savedRequest.redirectUrl)
    }
}

@SpringBootApplication
class DemoApplication

fun main(args: Array<String>) {
    runApplication<DemoApplication>(*args)
}
