% vis_mocapMovie()--visualizes mocap movie.
%
% Usage: vis_mocapMovie(inputMocapData, numFrameSkip)
%
% Input: inputMocapData--4x32x[nTimePoints] mocap data.
%        numFrameSkip--how many frames to skip in generating movie
%                      (subsampling). Larger the value, faster the movie.

% Hisotry
% 02/05/2020 Makoto. Support 129*X input format.
% 01/28/2020 Makoto. Created.

% Copyright (C) 2020, Makoto Miyakoshi
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 2 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1.07  USA

function vis_mocapMovie(inputMocapData, numFrameSkip)

% Convert raw data format to the processed data format. 02/05/2020 Makoto.
if size(inputMocapData,1) == 129
    inputMocapData = inputMocapData(1:128, :);
    inputMocapData = reshape(inputMocapData, [4 32 size(inputMocapData,2)]);
end

% Flip LR.
dimensionLR = 3;
originalCorrectedData_flipLR = inputMocapData;
originalCorrectedData_flipLR(dimensionLR,:,:) = inputMocapData(dimensionLR,:,:)*-1;

% Remove (0 0 0) points.
zeroCheckData = squeeze(sum(abs(originalCorrectedData_flipLR(1:3,:,:))));
zeroMask      = permute(repmat(zeroCheckData<10^-6, [1 1 4]), [3 1 2]);
originalCorrectedData_zeroReplaced = originalCorrectedData_flipLR;
originalCorrectedData_zeroReplaced(zeroMask) = NaN;
    

%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Define body parts. %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%
% Head polygon face definition.
headIdx = [5 6 7 5];

% Torso polygon face definition.
torsoIdx1 = [8 9 10 11 8];
torsoIdx2 = [8 9 21 8];
torsoIdx3 = [10 9 27 10];
torsoIdx4 = [9 21 27 9];
torsoIdx5 = [8 11 21 8];
torsoIdx6 = [10 11 27 10];
torsoIdx7 = [11 21 27 11];

% Hand polygon face definition.
handIdx   = [16:20 16];

% Foot polygon face definition.
footRIdx  = [23:26 23];
footLIdx  = [29:32 29];

% Non-rigid-body line definition.
armIdx   = [8 12 13 14 15 20];
kneeRIdx = [21 22 26];
kneeLIdx = [27 28 29];

    

% Obtain min-max coordinates in xyz.
minCoordinates = min(originalCorrectedData_zeroReplaced(1:3,:), [], 2);
maxCoordinates = max(originalCorrectedData_zeroReplaced(1:3,:), [], 2);


    
figure
view([0 0])
for frameIdx = 1:numFrameSkip:size(inputMocapData,3)
    cla
    
    hold on
    
    % Head.
    fill3(originalCorrectedData_zeroReplaced(1,headIdx,frameIdx), originalCorrectedData_zeroReplaced(3,headIdx,frameIdx), originalCorrectedData_zeroReplaced(2,headIdx,frameIdx), [0.66 0.76 1]);
    
    % Torso.
    fill3(originalCorrectedData_zeroReplaced(1,torsoIdx1,frameIdx), originalCorrectedData_zeroReplaced(3,torsoIdx1,frameIdx), originalCorrectedData_zeroReplaced(2,torsoIdx1,frameIdx), [0.66 0.76 1]);
    fill3(originalCorrectedData_zeroReplaced(1,torsoIdx2,frameIdx), originalCorrectedData_zeroReplaced(3,torsoIdx2,frameIdx), originalCorrectedData_zeroReplaced(2,torsoIdx2,frameIdx), [0.66 0.76 1]);
    fill3(originalCorrectedData_zeroReplaced(1,torsoIdx3,frameIdx), originalCorrectedData_zeroReplaced(3,torsoIdx3,frameIdx), originalCorrectedData_zeroReplaced(2,torsoIdx3,frameIdx), [0.66 0.76 1]);
    fill3(originalCorrectedData_zeroReplaced(1,torsoIdx4,frameIdx), originalCorrectedData_zeroReplaced(3,torsoIdx4,frameIdx), originalCorrectedData_zeroReplaced(2,torsoIdx4,frameIdx), [0.66 0.76 1]);
    fill3(originalCorrectedData_zeroReplaced(1,torsoIdx5,frameIdx), originalCorrectedData_zeroReplaced(3,torsoIdx5,frameIdx), originalCorrectedData_zeroReplaced(2,torsoIdx5,frameIdx), [0.66 0.76 1]);
    fill3(originalCorrectedData_zeroReplaced(1,torsoIdx6,frameIdx), originalCorrectedData_zeroReplaced(3,torsoIdx6,frameIdx), originalCorrectedData_zeroReplaced(2,torsoIdx6,frameIdx), [0.66 0.76 1]);
    fill3(originalCorrectedData_zeroReplaced(1,torsoIdx7,frameIdx), originalCorrectedData_zeroReplaced(3,torsoIdx7,frameIdx), originalCorrectedData_zeroReplaced(2,torsoIdx7,frameIdx), [0.66 0.76 1]);

    % Hand.
    fill3(originalCorrectedData_zeroReplaced(1,handIdx,frameIdx), originalCorrectedData_zeroReplaced(3,handIdx,frameIdx), originalCorrectedData_zeroReplaced(2,handIdx,frameIdx), [1 0.66 0.76]);

    % Feet.
    fill3(originalCorrectedData_zeroReplaced(1,footRIdx,frameIdx), originalCorrectedData_zeroReplaced(3,footRIdx,frameIdx), originalCorrectedData_zeroReplaced(2,footRIdx,frameIdx), [0.66 0.76 1]);
    fill3(originalCorrectedData_zeroReplaced(1,footLIdx,frameIdx), originalCorrectedData_zeroReplaced(3,footLIdx,frameIdx), originalCorrectedData_zeroReplaced(2,footLIdx,frameIdx), [0.66 0.76 1]);

    % Non-rigid bodies.
    plot3(originalCorrectedData_zeroReplaced(1,armIdx,frameIdx),   originalCorrectedData_zeroReplaced(3,armIdx,frameIdx),   originalCorrectedData_zeroReplaced(2,armIdx,frameIdx),   'color', [0 0 0], 'markerfacecolor', [0 0 0], 'marker', '.',  'linewidth', 1, 'markersize', 20);
    plot3(originalCorrectedData_zeroReplaced(1,kneeRIdx,frameIdx), originalCorrectedData_zeroReplaced(3,kneeRIdx,frameIdx), originalCorrectedData_zeroReplaced(2,kneeRIdx,frameIdx), 'color', [0 0 0], 'markerfacecolor', [0 0 0], 'marker', '.',  'linewidth', 1, 'markersize', 20);
    plot3(originalCorrectedData_zeroReplaced(1,kneeLIdx,frameIdx), originalCorrectedData_zeroReplaced(3,kneeLIdx,frameIdx), originalCorrectedData_zeroReplaced(2,kneeLIdx,frameIdx), 'color', [0 0 0], 'markerfacecolor', [0 0 0], 'marker', '.',  'linewidth', 1, 'markersize', 20);

    xlim([minCoordinates(1) maxCoordinates(1)])
    ylim([minCoordinates(3) maxCoordinates(3)])
    zlim([minCoordinates(2) maxCoordinates(2)])
    
    % Current time.
    title(sprintf('%.0f/%.0f (Frame)', frameIdx, size(inputMocapData,3)))
    
    drawnow
end