% torsoRollPitchYaw() - To compute roll, pitch, yaw of a torso rigid-body
%                       markers relative to room coordinate. A line connecting
%                       left and right shoulder markers is used for computing roll and yaw,
%                       while a line connecting mid-shoulder and mid-waste is used for pitch. 
%
% Use:
%   >> output = torsoRollPitchYaw(input)
%
% Inputs:
%   'input' - 3x6xt for xyz coordinates, 6 torso channels (marker indicecs 8, 9, 10, 11,
%             21, 27; must follow this order), and t frames.
%
% Outputs:
%   'output'   - 3xt for roll, pitch, and yaw, for all the t frames.

% Author:
%    Makoto Miyakoshi. SCCN, INC, UCSD. mmiyakoshi@ucsd.edu
%
% History:
%    02/01/2018 Makoto. Created. Switched from using rigid-body rotation t2x() to be based on room coordinates. 


function output = torsoRollPitchYaw(input)

% Check input.
if ~(size(input,1) == 3 & size(input,2) == 6)
    error('Input must be 3x6xt.');
end

% Convert single to double.
input = double(input);

%%%%%%%%%%%%%%%%%%%%%
%%% Compute roll. %%%
%%%%%%%%%%%%%%%%%%%%%
% Demean all the data to the shoulder line (use view([0 0]).
demeanedTemplate = bsxfun(@minus, input, mean(input(:, [1 3], :),2));

% Compute yaw angle in radian.
roll = atan(squeeze(demeanedTemplate(2,1,:))./squeeze(demeanedTemplate(3,1,:)))';

    % % Visualize it.
    % currentFrame = demeanedTemplate(:,:,frameIdx);
    % figure
    % scatter3(currentFrame(3,[2 4 5 6]), currentFrame(1,[2 4 5 6]), currentFrame(2,[2 4 5 6]), 'k')
    % hold on
    % scatter3(currentFrame(3,1), currentFrame(1,1), currentFrame(2,1), 'r')
    % scatter3(currentFrame(3,3), currentFrame(1,3), currentFrame(2,3), 'b')
    % xlim([-0.25 0.25])
    % ylim([-0.25 0.25])
    % line([0 0], ylim, 'color', [0 0 0])
    % line(xlim, [0 0], 'color', [0 0 0])
    % line([0 currentFrame(3,1)], [0 currentFrame(1,1)], 'color', [1 0 0])
    % view([0 0])

    
    
%%%%%%%%%%%%%%%%%%%%%%
%%% Compute pitch. %%%
%%%%%%%%%%%%%%%%%%%%%%
% Demean all the data to the line that connects the half-shoulder point and the half-waste point (use view([90 0])).
halfShoulderPoint = mean(input(:, [1 3], :),2);
halfWastePoint    = mean(input(:, [5 6], :),2);
halfHalfShoulderWastePoint = (halfShoulderPoint + halfWastePoint)/2;
demeanedTemplate  = bsxfun(@minus, input, halfHalfShoulderWastePoint);
halfShoulderPoint = mean(demeanedTemplate(:, [1 3], :),2);
halfWastePoint    = mean(demeanedTemplate(:, [5 6], :),2);

% Compute pitch angle in radian.
pitch = atan(squeeze(halfShoulderPoint(2,1,:))./abs(squeeze(halfShoulderPoint(1,1,:))))';

    % % Visualize it
    % currentFrame = demeanedTemplate(:,:,frameIdx);
    % figure
    % scatter3(currentFrame(3,[2 4 5 6]), currentFrame(1,[2 4 5 6]), currentFrame(2,[2 4 5 6]), 'k')
    % hold on
    % scatter3(currentFrame(3,1), currentFrame(1,1), currentFrame(2,1), 'r')
    % scatter3(currentFrame(3,3), currentFrame(1,3), currentFrame(2,3), 'b')
    % scatter3(halfShoulderPoint(3,1,frameIdx), halfShoulderPoint(1,1,frameIdx), halfShoulderPoint(2,1,frameIdx), 'g')
    % scatter3(halfWastePoint(3,1),    halfWastePoint(1,1),    halfWastePoint(2,1), 'g')
    % ylim([-0.25 0.25])
    % zlim([-0.25 0.25])
    % line([-0.25 0.25], [0 0], [0 0],'color', [0 0 0])
    % line([0 0], [-0.25 0.25], [0 0],'color', [0 0 0])
    % line([0 0], [0 0], [-0.25 0.25],'color', [0 0 0])
    % line([0 halfShoulderPoint(3,1,frameIdx)], [0 halfShoulderPoint(1,1,frameIdx)], [0 halfShoulderPoint(2,1,frameIdx)], 'color', [1 0 0])
    % view([90 0])


%%%%%%%%%%%%%%%%%%%%
%%% Compute yaw. %%%
%%%%%%%%%%%%%%%%%%%%
% Demean all the data to the shoulder line.
demeanedTemplate = bsxfun(@minus, input, mean(input(:, [1 3], :),2));

% Compute yaw angle in radian.
yaw = atan(squeeze(demeanedTemplate(1,1,:))./squeeze(demeanedTemplate(3,1,:)))';

    % % Visualize it
    % currentFrame = demeanedTemplate(:,:,frameIdx);
    % figure
    % scatter3(currentFrame(3,[2 4 5 6]), currentFrame(1,[2 4 5 6]), currentFrame(2,[2 4 5 6]), 'k')
    % hold on
    % scatter3(currentFrame(3,1), currentFrame(1,1), currentFrame(2,1), 'r')
    % scatter3(currentFrame(3,3), currentFrame(1,3), currentFrame(2,3), 'b')
    % xlim([-0.25 0.25])
    % ylim([-0.25 0.25])
    % line([0 0], ylim, 'color', [0 0 0])
    % line(xlim, [0 0], 'color', [0 0 0])
    % line([0 currentFrame(3,1)], [0 currentFrame(1,1)], 'color', [1 0 0])
    % view([0 90])
    
output = cat(1, roll, pitch, yaw);