% showSccnMovie() - Make stick figure animations for audiomaze.
%
% Usage: showSccnMovie(EEG, useCorrected, replaceZerosWithNan, applyLowPassFilter)
%
% Input: EEG -- EEGLAB EEG structure with EEG.etc.audiomaze. An example of
%               the data structure is shown below. 
%
%                                    maxMarker: [1x1 struct]
%                               phaseSpaceData: [129x241832 single]
%                               phaseSpaceTime: [1x241832 double]
%                                     fileplay: [1x1 struct]
%                                    trialInfo: [1x1 struct]
%                                   noisepitch: [1x1 struct]
%                                 trialResults: [1x1 struct]
%                                     headwall: [1x1 struct]
%                                handproximity: [1x1 struct]
%                                 behaviorData: [1x1 struct]
%                                      HEDtags: [1x1 struct]
%                              AudioCaptureWin: [1x1 struct]
%                          phaseSpaceCorrected: [4x32x241832 single]
%                    lessThanThreeLedMarkerIdx: [5x241832 logical]
%                 lessThanThreeLedMarkerLabels: {'Head'  'Torso'  'Hand'  'footR'  'footL'}
%
%        useCorrected -- [0|1] Whether to use the raw data stored  
%                        EEG.etc.audiomaze.phaseSpaceData [0] or 
%                        use the corrected data stored 
%                        EEG.etc.audiomaze.phaseSpaceData [0]
% replaceZerosWithNan -- [0|1] Whether to represent missing data as 0 [0]
%                        (which shows these data located in the center
%                        of the room at the floor height) or make them
%                        invisible in the movie [1].
% applyLowPassFilter  -- [0|1] Whether to show raw data [0] or show data after
%                        applying default pop_eegfiltnew() with 1-Hz
%                        pass-band edge low-pass filter (FIR, Hamming, -6dB @ 0.5 Hz,
%                        TBW 1 Hz) [1].

% History:
% 02/02/2018 Makoto. Made into a function.
% 12/20/2017 Makoto. Updated.
% 09/12/2017 CL. Reverted view angle to be consistent across all movies
% 09/08/2017 Makoto. Modified John's code to make it work.
% 07/26/2017 CL. Turned off view interpolation for constant top down view
% 08/28/2017 CL. Interp on, based on length of movie. Use rigid body corrected PS data from EEG.etc.
% 07/25/2017 John Iversen. Created.



function showSccnMovie(EEG, useCorrected, replaceZerosWithNan, applyLowPassFilter)

% Set path to dependent files.
currentFunctionName = mfilename;
fullPath            = mfilename('fullpath');
currentFolder       = fullPath(1:end-length(currentFunctionName));
pathToAdd           = [currentFolder 'showSccnMovieDependency'];
addpath(pathToAdd);
disp(['Path is set to the dependency folder ' pathToAdd '.'])

% Disable movie saving function.
saveMovie = 0;

% Obtain subject name and maze type.
subjRun = EEG.filename(5:end-4);
maze    = subjRun(end-1);
run     = subjRun(end);
    
% Obtain LSL time stamps.
MOCAP.times = EEG.etc.audiomaze.phaseSpaceTime;

% Select raw or corrected data to show movie.
if     useCorrected == 0
    tmp = EEG.etc.audiomaze.phaseSpaceData;
    tmp = reshape(tmp(1:128,:), [4 32 size(tmp,2)]);
    nanMask = tmp==0;
    tmp(nanMask) = NaN;
    rbCorrectedPS = tmp(1:3,:,:);
elseif useCorrected == 1
    rbCorrectedPS = EEG.etc.audiomaze.phaseSpaceCorrected(1:3,:,:);
end

% Select showing missing data with zeros or NaNs.
if replaceZerosWithNan == 1
    nanMask = (rbCorrectedPS==0);
    rbCorrectedPS(nanMask) = NaN;
end

% Select whether appling 1-Hz low-pass filter.
if applyLowPassFilter == 1
    cutOffFreqHz    = 1;
    srate           = evalin('base', 'EEG.srate');
    EEG1HzLpf       = eeg_emptyset();
    EEG1HzLpf.data  = reshape(rbCorrectedPS, [size(rbCorrectedPS,1)*size(rbCorrectedPS,2) size(rbCorrectedPS,3)]);
    EEG1HzLpf.srate = srate;
    EEG1HzLpf.pnts  = size(EEG1HzLpf.data,2);
    EEG1HzLpf       = pop_eegfiltnew(EEG1HzLpf, 0, cutOffFreqHz);
    rbCorrectedPS   = reshape(EEG1HzLpf.data, size(rbCorrectedPS));
end

% Permute the dimension order.
rbCorrectedPS = permute(rbCorrectedPS,[1,3,2]);
MOCAP.interpMarkerCoords = rbCorrectedPS([3 1 2],:,:);

% Select whether saving this movie or not.
if saveMovie
    disp('writing video!')
    if isunix
        mov = VideoWriter(subjRun);
    elseif ispc
        mov = VideoWriter(subjRun,'MPEG-4');
    end
    open(mov);
end

% Define the mocap markers.
X = defineMocapMarkers;
markers = X.mocap.markers;

%% Parse HED tags.
HED = EEG.etc.audiomaze.HEDtags;
proximity = [];
touch = [];
for i = 1:length(HED.time_series)
    if strfind(HED.time_series{i},'HandProximity/Onset')
        proximity(end+1,1) = HED.time_stamps(i);
    elseif strfind(HED.time_series{i},'HandProximity/Offset')
        proximity(end,2) = HED.time_stamps(i);
    end
    
    if strfind(HED.time_series{i},'HandThroughWall/Onset')
        touch(end+1,1) = HED.time_stamps(i);
    elseif strfind(HED.time_series{i},'HandThroughWall/Offset')
        touch(end,2) = HED.time_stamps(i);
    end
    
end
if isempty(touch)
    touch = [inf -inf]; %no time will match
end

%animate the stick figure

%% define animation step
if saveMovie
    step = 51; % appears ~realtime on screen, sped up nicely on movie
else
    step = 102; %appears about same rate as movie will with smaller step (~3x speed up)
end
step_s = step/480; %step in samples

%% adjust touch times so we don't skip over any
proximity(:,1) = proximity(:,1) - step_s;
proximity(:,2) = proximity(:,2) + step_s;
touch(:,1) = touch(:,1) - step_s;
touch(:,2) = touch(:,2) + step_s;

%%
%figure
%cla
%set(gca,'ydir','reverse')
plotMaze(maze)
set(gca, 'XTick', []); set(gca, 'YTick', []); set(gca, 'ZTick', [])
set(gca, 'position', [0.2 0.2 0.6 0.6]);
set(figure(11), 'Position', [1600 0 800 800], 'Color', [0.66 0.76 1])

%% make a view trajectory
% each row of this provides keyframes for the view [time(sec) azimuth elevation]
% I did a certain amount of hand tuning to get it to look good--depends on
% the maze, I expect.
viewKeyframes = [0 0 90; 20 0 90; 45 4 59; 85 64 23; 120 64 23; 155 12 41; MOCAP.times(end) 12 41]; %time, az, el
%  viewKeyframes = [0 0 90; MOCAP.times(end)/6 0 90; MOCAP.times(end)*2/6 4 59;...
%  MOCAP.times(end)*3/6 64 23; MOCAP.times(end)*4/6 64 23; MOCAP.times(end)*5/6 12 41; MOCAP.times(end) 12 41]; %time, az, el

az = interp1(viewKeyframes(:,1)*480, viewKeyframes(:,2), MOCAP.times*480,'pchip',viewKeyframes(end,2));
el = interp1(viewKeyframes(:,1)*480, viewKeyframes(:,3), MOCAP.times*480,'pchip',viewKeyframes(end,3));

%check the trajectory
%figure
%plot(MOCAP.times*480,az,MOCAP.times*480,el)

%% make the movie
h=[];
hs=[];
for t = 1:step:length(MOCAP.times)-480
    %for t = 1:step:75000 %debug, just part of the trial
    idx = t+[0:step-1];
    idx(idx>length(MOCAP.times))=[];
    points = squeeze(nanmean(MOCAP.interpMarkerCoords(:,idx,:),2));
    delete([h hs])
    h=[];  % Added by Makoto. 09/08/2017.
    hs=[]; % Added by Makoto. 09/08/2017.
    %plot the stick f igure
    h = plotStick(points);
    xlabel('');ylabel('');zlabel('')
    %plot a sphere if a wall touch
    handpos = mean(points(:,markers.rightHand),2);
    if any(MOCAP.times(t)>proximity(:,1) & MOCAP.times(t)<=proximity(:,2))
        if any(MOCAP.times(t)>touch(:,1) & MOCAP.times(t)<=touch(:,2))
            hs=plotsphere(handpos,.2,'color','r','nvert',15); %through wall
        else
            hs=plotsphere(handpos,.15,'color',[1 .5 0],'nvert',15); %wall proximity
        end
    end
    axis([-3 3 -3 3 0 3])
    view(az(t),el(t))
    title(sprintf('Subj%s Run %s%d, Time:%6.1f sec', EEG.subject, maze, str2num(run), MOCAP.times(t)), 'fontsize', 16)
    drawnow
    
    if saveMovie
        currFrame = getframe(11);
        writeVideo(mov,currFrame);
    end
end

if saveMovie
    close(mov);
    disp('done')
end