% rigidBodyTransformation() - performs rigid body transformation.
%
% Use  : [fitResult, rmsError, rotationMatrix, centroidOffset] = rigidBodyTransformation(destination, template)
%
% Input: destination --the 3xn matrix (e.g., x-, y-, and z-coordinates) to
%                      which the templated is best fit. 
%        dataToFit   --the 3xn matrix (e.g., x-, y-, and z-coordinates)
%                      which will be moved to the destination.
%
% Output: fitResult     --the 3xn matrix (e.g., x-, y-, and z-coordinates)
%                         which is the result of optimal rigid-body rotation and fitting.
%         rmsError      --nx1 matrix that represents rmsError across channels.
%         rotationMatrix--rotation matrix.
%         centroidOffset--3x1 vector (e.g., x-, y-, and z-coordinates) that represents centroid offsets.
%
% Rotation using SVD by Jan Simon: https://www.mathworks.com/matlabcentral/answers/3862-fit-rigid-body-to-points

% Author: Makoto Miyakoshi, SCCN, INC, UCSD.
%
% History:
%   01/23/2020 Makoto. Added some comments and tests.
%   01/26/2018 Makoto. Updated to detect commonly available (and >=3) channels between destination and dataToFit. This change may be too late but it will be useful anyway.
%   08/04/2017 Makoto. Created.

function [fitResult, rmsError, rotationMatrix, centroidOffset] = rigidBodyTransformation(destination, dataToFit)

% Detect common channels available.
availableChannels1 = find(~isnan(destination(1,:)));
availableChannels2 = find(~isnan(dataToFit(1,:)));
commonlyAvailableChannels = intersect(availableChannels1,availableChannels2);
if length(commonlyAvailableChannels)<3
    error('Less than 3 channels available.')
end

% De-mean inputs.
destinationDash = bsxfun(@minus, destination(:,commonlyAvailableChannels), mean(destination(:,commonlyAvailableChannels),2));
dataToFitDash   = bsxfun(@minus, dataToFit(:,commonlyAvailableChannels),   mean(dataToFit(:,commonlyAvailableChannels),2));
    % destinationBar = R*dataToFitBar;

% Compute the rotation matrix.
R = destinationDash * dataToFitDash' / size(dataToFitDash,2);
[U, S, V] = svd(R);
if det(U * V) > 0 % This part is probably from Challis (1995) to address the issue of reflection.
    rotationMatrix = U * transpose(V);
else
    rotationMatrix = U * [1,0,0; 0,1,0; 0,0,-1] * transpose(V);
end

% % Test my calculation (01/23/2020)
% rotatinMatrix2 = destinationDash*pinv(dataToFitDash')';
% rotationMatrix
% rotatinMatrix2

% Compute rigid-body rotation.
rotatedData = rotationMatrix*dataToFitDash;

% Compute rigid-body shift.
centroidOffset = mean(destination(:,commonlyAvailableChannels),2) - mean(rotatedData,2);
fitResultTruncated = bsxfun(@plus, rotatedData, centroidOffset);
fitResult          = nan(size(dataToFit));
fitResult(:,commonlyAvailableChannels) = fitResultTruncated;

% Taken from icp.m: where p1 and p2 are 3xn matrices.
dsq      = sum(power(destination(:,commonlyAvailableChannels) - fitResult(:,commonlyAvailableChannels), 2),1);
rmsError = sqrt(mean(dsq));