% rigidBodyTransformation() - performs rigid body transformation.
%
% Use  : [fitResult, rmsError, rotationMatrix, centroidOffset] = rigidBodyTransformation(destination, dataToFit)
%
% Input: destination --the 3xn matrix (e.g., x-, y-, and z-coordinates) to
%                      which the templated is best fit. 
%        dataToFit   --the 3xn matrix (e.g., x-, y-, and z-coordinates)
%                      which will be moved to the destination.
%
% Output: fitResult     --the 3xn matrix (e.g., x-, y-, and z-coordinates)
%                         which is the result of optimal rigid-body rotation and fitting.
%         rmsError      --nx1 matrix that represents rmsError across channels.
%         rotationMatrix--rotation matrix.
%         centroidOffset--3x1 vector (e.g., x-, y-, and z-coordinates) that represents centroid offsets.
%
% Rotation using SVD by Jan Simon: https://www.mathworks.com/matlabcentral/answers/3862-fit-rigid-body-to-points

% Author: Makoto Miyakoshi, SCCN, INC, UCSD.
%
% History:
%   01/28/2020 Makoto. Cleaned.
%   01/23/2020 Makoto. Added some comments and tests.
%   01/26/2018 Makoto. Updated to detect commonly available (and >=3) channels between destination and dataToFit. This change may be too late but it will be useful anyway.
%   08/04/2017 Makoto. Created.

function [fitResult, rmsError, rotationMatrix, centroidOffset] = rigidBodyTransformation(destination, dataToFit)

% De-mean inputs.
destinationDash = bsxfun(@minus, destination, mean(destination,2));
dataToFitDash   = bsxfun(@minus, dataToFit,   mean(dataToFit,2));
    % destinationBar = R*dataToFitBar;

% Compute the rotation matrix.
R = destinationDash * dataToFitDash' / size(dataToFitDash,2);
[U, S, V] = svd(R);
if det(U * V) > 0 % This part is probably from Challis (1995) to address the issue of reflection.
    rotationMatrix = U * transpose(V);
else
    rotationMatrix = U * [1,0,0; 0,1,0; 0,0,-1] * transpose(V);
end

% % Test my calculation (01/23/2020)
% rotatinMatrix2 = destinationDash*pinv(dataToFitDash')';
% rotationMatrix
% rotatinMatrix2

% Compute rigid-body rotation.
rotatedData = rotationMatrix*dataToFitDash;

% Compute rigid-body shift.
centroidOffset = mean(destination,2) - mean(rotatedData,2);
fitResult      = bsxfun(@plus, rotatedData, centroidOffset);


% Taken from icp.m: where p1 and p2 are 3xn matrices.
dsq      = sum(power(destination - fitResult, 2),1);
rmsError = sqrt(mean(dsq));