% pairedChannelCorrection() - Detect outliers in distance between a pair of
%                             channels in following critera: 1) More than
%                             n-times of median distance across time, and
%                             2) high-frequency spike noise in distance.
%                             
%
% Use:
%   >> channelCorrected = pairedChannelCorrection(channelToBeReferenced, channelToBeCorrected, medianDistanceFactor, cutOffFreqHz)
%
% Inputs:
%   'channelToBeReferenced'- 3xn double. Matrix of xyz coordinates for n time points. Should be a
%                            channel of either a part of ridig body or already corrected 
%                            with this function. Will be 5-Hz low-pass filtered.                           filtered 
%   'channelToBeCorrected' - 3xn double. Matrix of xyz coordinates for n time points. Output will
%                            be this data with correction.
%   'medianDistanceFactor' - 1x1 double. Detects outlier in distance in SD.
%                            Recommended default value is 2.
%   'cutOffFreqHz'         - 1x1 double [Hz]. Used for high/low-pass filter.
%
% Outputs:
%   'channelCorrected'     - 3xn double. Matrix of corrected xyz coordinates.

% Author:
%    Makoto Miyakoshi. SCCN, INC, UCSD. mmiyakoshi@ucsd.edu
%
% History:
%    01/17/2018 Makoto. Removed spike correction.
%    12/27/2017 Makoto. Updated.
%    08/25/2017 Makoto. Created.

% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 2 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

function channelCorrected = pairedChannelCorrection(channelToBeReferenced, channelToBeCorrected, medianDistanceFactor, cutOffFreqHz)
          
% Detect initial and ending NaN chunks in channelToBeCorrected.
[initialNanFrameIdx, endingNanFrameIdx] = findInitialAndEndingNanFrames(channelToBeCorrected(1,:), 1);
nonNanFrameIdx = setdiff(1:size(channelToBeCorrected,2), [initialNanFrameIdx endingNanFrameIdx]);

% Interpolate NaN in channelToBeCorrected.
if any(find(isnan(channelToBeCorrected(1,nonNanFrameIdx))))
    channelToBeCorrected = customInterpolation(channelToBeCorrected, 1);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%                          
%%% Remove datapoints with median*medianDistanceFactor distance from low-passed corrected data. %%%                          
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%                          
channelCorrectedTruncated      = channelToBeCorrected(nonNanFrameIdx);
channelToBeReferencedTruncated = channelToBeReferenced(nonNanFrameIdx);
srate       = evalin('base', 'EEG.srate');
filterOrder = pop_firwsord('hamming', srate, cutOffFreqHz);

% Apply low-pass filter for 'channelToBeReferenced'.
EEG_Lpf       = eeg_emptyset();
EEG_Lpf.data  = channelToBeReferencedTruncated;
EEG_Lpf.srate = srate;
EEG_Lpf.pnts  = size(channelToBeReferencedTruncated,2);
EEG_Lpf       = pop_eegfiltnew(EEG_Lpf, 0, cutOffFreqHz, filterOrder);

% Detect outlier Calculate distance.
marker1            = EEG_Lpf.data;
marker2            = channelCorrectedTruncated;
pairMarkerDistanceTruncated = sqrt(sum((marker1-marker2).^2))*100; % Unit is cm.

% Detect outliers in distance changes: n x median value.
medianValue           = nanmedian(pairMarkerDistanceTruncated);
wrongDistanceFrameIdx = find(pairMarkerDistanceTruncated>medianValue*medianDistanceFactor);

% Skip if nothing to correct.
if isempty(wrongDistanceFrameIdx)
    channelCorrected = channelToBeCorrected;
    return
end

% Create NaN mask.
nanMask = zeros(length(pairMarkerDistanceTruncated),1);
nanMask(wrongDistanceFrameIdx) = 1;
nanMask = logical(nanMask);

% Interpolate NaN mask.
currentDataNanMasked = channelToBeCorrected;
currentDataNanMasked(:,nanMask) = NaN;
channelCorrectedTruncated = customInterpolation(currentDataNanMasked, cutOffFreqHz);

% Prepare output.
channelCorrected = nan(size(channelToBeCorrected));
channelCorrected(:,nonNanFrameIdx) = channelCorrectedTruncated;