% headRollPitchYaw() - To compute roll, pitch, yaw of a head rigid-body
%                      markers relative to room coordinate. A line connecting
%                      left and right head markers is used for computing roll and yaw,
%                      while a mid-channel midline coordinate is used for pitch. 
%
% Use:
%   >> output = headRollPitchYaw(input)
%
% Inputs:
%   'input'  - 3x7xt for xyz coordinates, 7 head channels (marker indicecs
%              1:7; must follow this order), and t frames.
%
% Outputs:
%   'output' - 3xt for roll, pitch, and yaw, for all the t frames.

% Author:
%    Makoto Miyakoshi. SCCN, INC, UCSD. mmiyakoshi@ucsd.edu
%
% History:
%    02/01/2018 Makoto. Created. Switched from using rigid-body rotation t2x() to be based on room coordinates. 


function output = headRollPitchYaw(input)

% Check input.
if ~(size(input,1) == 3 & size(input,2) == 7)
    error('Input must be 3x7xt.');
end

% Convert single to double.
input = double(input);

%%%%%%%%%%%%%%%%%%%%%
%%% Compute roll. %%%
%%%%%%%%%%%%%%%%%%%%%
% Demean all the data to the ear-to-ear line (use view([0 0]).
demeanedTemplate = bsxfun(@minus, input, mean(input(:, [5 6], :),2));

% Compute yaw angle in radian.
roll = atan(squeeze(demeanedTemplate(2,5,:))./squeeze(demeanedTemplate(3,5,:)))';

    % % Visualize it.
    % frameIdx = 5000
    % currentFrame = demeanedTemplate(:,:,frameIdx)
    % rad2deg(roll(frameIdx))
    % figure
    % scatter3(currentFrame(3,[1:4 7]), currentFrame(1,[1:4 7]), currentFrame(2,[1:4 7]), 'k')
    % hold on
    % scatter3(currentFrame(3,5), currentFrame(1,5), currentFrame(2,5), 'r')
    % scatter3(currentFrame(3,6), currentFrame(1,6), currentFrame(2,6), 'b')
    % xlim([-0.15 0.15])
    % ylim([-0.15 0.15])
    % zlim([-0.15 0.15])
    % line([-0.15 0.15], [0 0], [0 0], 'color', [0 0 0])
    % line([0 0], [-0.15 0.15], [0 0], 'color', [0 0 0])
    % line([0 0], [0 0], [-0.15 0.15], 'color', [0 0 0])
    % line([0 currentFrame(3,5)], [0 currentFrame(1,5)], [0 currentFrame(2,5)], 'color', [1 0 0])
    % view([0 0])

    
    
%%%%%%%%%%%%%%%%%%%%%%
%%% Compute pitch. %%%
%%%%%%%%%%%%%%%%%%%%%%
% Demean all the data to the line that connects the mid 3-4 point and the mid 5-6 point (use view([90 0])).
mid34Point   = mean(input(:, [3 4], :),2);
mid56Point   = mean(input(:, [5 6], :),2);
mid3456Point = (mid34Point + mid56Point)/2;
demeanedTemplate  = bsxfun(@minus, input, mid3456Point);
mid34Point   = mean(demeanedTemplate(:, [3 4], :),2);
mid56Point   = mean(demeanedTemplate(:, [5 6], :),2);

% Compute pitch angle in radian.
pitch = atan(squeeze(mid34Point(2,1,:))./abs(squeeze(mid34Point(1,1,:))))';

    % % Visualize it
    % currentFrame = demeanedTemplate(:,:,frameIdx);
    % figure
    % scatter3(currentFrame(3,:), currentFrame(1,:), currentFrame(2,:), 'k')
    % hold on
    % scatter3(mid34Point(3,1,frameIdx), mid34Point(1,1,frameIdx), mid34Point(2,1,frameIdx), 'g')
    % scatter3(mid56Point(3,1,frameIdx), mid56Point(1,1,frameIdx), mid56Point(2,1,frameIdx), 'g')
    % ylim([-0.15 0.15])
    % zlim([-0.15 0.15])
    % line([-0.15 0.15], [0 0], [0 0],'color', [0 0 0])
    % line([0 0], [-0.15 0.15], [0 0],'color', [0 0 0])
    % line([0 0], [0 0], [-0.15 0.15],'color', [0 0 0])
    % line([0 mid34Point(3,1,frameIdx)], [0 mid34Point(1,1,frameIdx)], [0 mid34Point(2,1,frameIdx)], 'color', [1 0 0])
    % view([90 0])
    % rad2deg(pitch(frameIdx))

%%%%%%%%%%%%%%%%%%%%
%%% Compute yaw. %%%
%%%%%%%%%%%%%%%%%%%%
% Demean all the data to the ear-to-ear line.
demeanedTemplate = bsxfun(@minus, input, mean(input(:, [5 6], :),2));

% Compute yaw angle in radian.
yaw = atan(squeeze(demeanedTemplate(1,5,:))./squeeze(demeanedTemplate(3,5,:)))';

    % frameIdx = 9000
    % % Visualize it.
    % currentFrame = demeanedTemplate(:,:,frameIdx);
    % 
    % rad2deg(roll(frameIdx))
    % 
    % figure
    % scatter3(currentFrame(3,[1:4 7]), currentFrame(1,[1:4 7]), currentFrame(2,[1:4 7]), 'k')
    % hold on
    % scatter3(currentFrame(3,5), currentFrame(1,5), currentFrame(2,5), 'r')
    % scatter3(currentFrame(3,6), currentFrame(1,6), currentFrame(2,6), 'b')
    % xlim([-0.15 0.15])
    % ylim([-0.15 0.15])
    % zlim([-0.15 0.15])
    % line([-0.15 0.15], [0 0], [0 0], 'color', [0 0 0])
    % line([0 0], [-0.15 0.15], [0 0], 'color', [0 0 0])
    % line([0 0], [0 0], [-0.15 0.15], 'color', [0 0 0])
    % line([0 currentFrame(3,5)], [0 currentFrame(1,5)], [0 currentFrame(2,5)], 'color', [1 0 0])
    % view([0 90])
    
output = cat(1, roll, pitch, yaw);