% findInitialAndEndlingNanFrames() - Returns frame indices of initial and
%                                    ending NaN chunk that is not
%                                    recoverable with rigid body fitting
%                                    (i.e., less than 3 markers available).  
%
% Use:
%   >> [initialNanFrameIdx, endingNanFrameIdx] = findInitialAndEndlingNanFrames(nanInputCoordinaes, minAvailableChannel);
%
%
% Inputs:
%   'nanInputCoordinaes'  - 3xnxm double. Tensor of xyz coordinates,
%                           n channels, m frames. NaN means missind data.
%   'minAvailableChannel' - 1x1 minimum number of channel to be acceptable.
%                           For rigid-body fitting, 3. For single-channel
%                           nan-chunk detection, 0.
%
% Outputs:
%   'initialNanFrameIdx' - kx1 double. Frame indices that contains 1.
%   'endingNanFrameIdx'  - lx1 double. Frame indices that contains the
%                          final frame of the data.

% Author:
%    Makoto Miyakoshi. SCCN, INC, UCSD. mmiyakoshi@ucsd.edu
%
% History:
%    01/20/2018 Makoto. Debugged the ending NaN chunk process.
%    01/18/2018 Makoto. Created.

function [initialNanFrameIdx, endingNanFrameIdx] = findInitialAndEndingNanFrames(nanInputCoordinates, minAvailableChannel)

% Find out unrecoverable frames with rigid body correction.
if length(size(nanInputCoordinates)) == 2 & minAvailableChannel == 1
    unrecoverbleFrameIdx = find(isnan(nanInputCoordinates));
else
    unrecoverbleFrameIdx = find(sum(isnan(squeeze(nanInputCoordinates(1,:,:)))) > size(nanInputCoordinates,2)-minAvailableChannel);
end

%{

% This is for testing.
unrecoverbleFrameIdx(1:20)       = 1:20;
unrecoverbleFrameIdx(end-20:end) = length(nanInputCoordinates)-20:length(nanInputCoordinates);

%}

% Find frames of in the initial nan chunk.
if find(ismember(unrecoverbleFrameIdx,1))
    diffDiffIdx = find(diff(diff(unrecoverbleFrameIdx)));
    if isempty(diffDiffIdx)
        diffDiffIdx = length(unrecoverbleFrameIdx)-1;
    end
    initialNanFrameIdx = 1:unrecoverbleFrameIdx(diffDiffIdx(1)+1);
else
    initialNanFrameIdx = [];
end

% Find frames of in the ending nan chunk.
if find(ismember(unrecoverbleFrameIdx, length(nanInputCoordinates)))
    diffDiffIdx = find(diff(diff(unrecoverbleFrameIdx)));
    if isempty(diffDiffIdx)
        diffDiffIdx = 0;
    end
    endingNanFrameIdx = unrecoverbleFrameIdx(diffDiffIdx(end)+1:end);
else
    endingNanFrameIdx = [];
end