% distanceMatrixOutlierDetection() - Detects outliers in frames of distance
%                                    matrices. 3SD deviation from median
%                                    across time will be detected.
%
% Use:
%   >> distanceOutlierIdx = distanceMatrixOutlierDetection(inputCoordinates)
%
% Inputs:
%   'inputCoordinates'   - 3xjxk double. 3 is PhaseSpace xyz, j is channels,
%                           and k is data points.
%   'distanceSd'         - 1x1 double.
%
% Outputs:
%   'distanceOutlierIdx' - jxk double. j is channels, and k is data points.

% Author:
%    Makoto Miyakoshi. SCCN, INC, UCSD. mmiyakoshi@ucsd.edu
%
% History: 
%    09/11/2017. Makoto. Created.

% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 2 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

function distanceOutlierIdx = distanceMatrixOutlierDetection(inputCoordinates, distanceSd)

% Compute distance matrix.
distanceMatrix = zeros(size(inputCoordinates,2), size(inputCoordinates,2), size(inputCoordinates,3));
for row = 1:size(inputCoordinates,2)
    for column = 1:size(inputCoordinates,2)
        if row >= column
            continue
        end
        distanceMatrix(row, column, :) = squeeze(sum(bsxfun(@minus, squeeze(inputCoordinates(:,row,:)), squeeze(inputCoordinates(:,column,:))).^2));
    end
end
        % figure; imagesc(mean(distanceMatrix,3))
        
% Compute 2SD using median for robustness.
distanceMatrixDemedian = bsxfun(@minus, distanceMatrix, median(distanceMatrix,3));
distanceMatrixStd      = std(distanceMatrixDemedian,0,3);

% Detect outliers
outlierMatrix = bsxfun(@gt, distanceMatrixDemedian, distanceMatrixStd*distanceSd);

% Summarize results into channels.
distanceOutlierIdx = squeeze(sum(outlierMatrix,1))+squeeze(sum(outlierMatrix,2));