% customSpikeDetection() - High-pass filter data, apply threshold, detect high-freq
%                          spikes, extend the window, replace high-freq high-amp with NaN.
%
% Use:
%   >> spikeReplacedWithNaN = customSpikeDetection(dataWithSpikes, hpfCutoffHz, thresholdSd, pointSpreadInFrame)
%
% Inputs:
%   'dataWithSpikes'     - 3xjxk double. 3 is PhaseSpace xyz, j is channels,
%                          and k is data points. Data with spikes.
%   'hpfCutoffHz'        - 1x1 double [Hz]. TBW is same as the cutoff frequency.
%   'thresholdSd'        - 1x1 double. High-freq spike detection threshold
%                          in SD. Recommended default is 8.
%   'pointSpreadInFrame' - 1x1 double [Frame]. Extend the rejection
%                          window +/- half of the value. If odd number, +1.
%
% Outputs:
%   'spikeReplacedWithNaN' - 3xjxk double. Spikes are replaced with NaN.

% Author:
%    Makoto Miyakoshi. SCCN, INC, UCSD. mmiyakoshi@ucsd.edu
%
% History: 
%    08/25/2017. Makoto. Created.

% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 2 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

function spikeReplacedWithNaN = customSpikeDetection(dataWithSpikes, hpfCutoffHz, thresholdSd, pointSpreadInFrame)

% If data are multi-channel, reshape data into two-demensional.
if     length(size(dataWithSpikes)) == 2
    inputData = dataWithSpikes;
elseif length(size(dataWithSpikes)) == 3
    inputData = reshape(dataWithSpikes, [size(dataWithSpikes,1)*size(dataWithSpikes,2), size(dataWithSpikes,3)]);
end

% Generate data frame index for interpolation.
dataFrameIdx = 1:size(inputData,2);

% Calculate filter order.
srate       = evalin('base', 'EEG.srate');
filterOrder = pop_firwsord('hamming', srate, hpfCutoffHz);

% Detect spikes in each channel.
spikeReplacedWithNaN = zeros(size(inputData));
for chIdx = 1:size(inputData,1)
    
    % Obtain the current single-channel time series.
    currentData  = inputData(chIdx,:);
    
    % Perform high-pass filter.
    hpfData       = eeg_emptyset();
    hpfData.data  = currentData;
    hpfData.srate = evalin('base', 'EEG.srate');
    hpfData.pnts  = length(currentData);
    hpfData       = pop_eegfiltnew(hpfData, hpfCutoffHz, 0, filterOrder);
    hpfData       = hpfData.data;
    dataStd       = std(hpfData);
    
    % Detect high-frequency spikes.
    hpfOutlierIdx = find(hpfData < -thresholdSd*dataStd | hpfData > thresholdSd*dataStd);
    
    % Extend the frames.
    pointSpreadValues = floor(-pointSpreadInFrame/2):ceil(pointSpreadInFrame/2);
    extendedFrames    = [];
    for extensionFrameIdx = 1:length(pointSpreadValues)
        extendedFrames = [extendedFrames hpfOutlierIdx+pointSpreadValues(extensionFrameIdx)];
    end
    extendedFrames = sort(unique(extendedFrames));
    hpfOutlierIdx  = extendedFrames(extendedFrames>=1 & extendedFrames<=length(currentData));
    
    % Create NaN mask.
    nanMask = zeros(length(hpfData),1);
    nanMask(hpfOutlierIdx) = 1;
    nanMask = logical(nanMask);
    
    % Apply NaN mask.
    currentDataNanMasked = currentData;
    currentDataNanMasked(nanMask) = NaN;
    
    % Store the result.
    spikeReplacedWithNaN(chIdx,:) = currentDataNanMasked;
end

% If data are multi-channel, reshape data back into three-demensional.
if     length(size(dataWithSpikes)) == 2
    spikeReplacedWithNaN = spikeReplacedWithNaN;
elseif length(size(dataWithSpikes)) == 3
    spikeReplacedWithNaN = reshape(spikeReplacedWithNaN, size(dataWithSpikes));
end