% 02/12/2020 Makoto. Created.

% Perform arm and knee correction.
clear
clc

sdThreshold = 2;
addpath /data/mobi/Audiomaze/makoto/code/phaseSpaceCorrectionTools

% Define rigid body LED marker indices.
headIdx   = [1:7];
torsoIdx  = [8:11 21 27];
handIdx   = [16:20];
footR_Idx = [23:26];
footL_Idx = [29:32];
armIdx    = [12:15];
kneeIdx   = [22 28];

allSets = dir('/data/mobi/Audiomaze/makoto/p05_neuralNetworkForNonRigidBodies/*.set');
for setIdx = 101:length(allSets) % 819_C3 (idx85) has no right knee.
    
    % Load .set file.
    loadingName = allSets(setIdx).name;
    EEG         = pop_loadset('filename', loadingName, 'filepath', '/data/mobi/Audiomaze/makoto/p05_neuralNetworkForNonRigidBodies', 'loadmode', 'info');

    % Convert zeros into NaN.
    phaseSpaceData = EEG.etc.audiomaze.phaseSpaceCorrected;
    phaseSpaceData(phaseSpaceData==0) = NaN; % This is necessary, DO NOT disable (02/13/2020 Makoto).
%     vis_mocapMovie(phaseSpaceData, 48)
    


    % Replace zeros with NaNs if all xyz are zeros. (02/10/2020 Makoto)
    torsoData     = phaseSpaceData(1:3, torsoIdx, :);
        % % Test.
        % torsoData(:,3, 1234560)=0;
        % torsoData(:,4, 1234565)=0;
        % torsoData(:,5, 1234570)=0;
    zeroMask = squeeze(sum(torsoData==0));
    zeroXyzIdx = find(zeroMask==3);
    [chIdx, timeIdx] = ind2sub(size(zeroMask), zeroXyzIdx);
    torsoData(:,chIdx,timeIdx) = NaN;

    % Detect all-NaN torso frame indices to remove.
    tmp = reshape(torsoData, [size(torsoData,1)*size(torsoData,2), size(torsoData,3)]);
    tmp2 = isnan(tmp);
    tmp3 = sum(tmp2);
    allNanTorsoIdx = find(tmp3==18);
    %nanChannelVec = squeeze(sum(isnan(phaseSpaceData(1,:,:)),2));
    %allNanIdx     = find(nanChannelVec == size(phaseSpaceData,2));
    torsoNonNanFrameIdx = setdiff(1:size(torsoData,3), allNanTorsoIdx);
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %%% Rotate all the frames into the first frame using torso rigid body rotation. %%%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    wholeBodyRotated   = nan(3, size(phaseSpaceData,2), size(phaseSpaceData,3));
    rotationMatrixList = nan(3,3,size(torsoData,3));
    for goodFrameIdxIdx = 1:length(torsoNonNanFrameIdx)
        if mod(goodFrameIdxIdx, 10000) == 0
            disp(sprintf('%.0f/%.0f...', goodFrameIdxIdx, length(torsoNonNanFrameIdx)));
        end
        frameIdx = torsoNonNanFrameIdx(goodFrameIdxIdx);
        
        % Fix NaN for one of the xyz coordinates that happened to be genuinly zero, 
        currentTorsoData = torsoData(:,:,frameIdx);
        if any(isnan(currentTorsoData(:)))
            xyzNanCheck = find(sum(isnan(currentTorsoData)) == 3);
            if isempty(xyzNanCheck)
                nanMask = isnan(currentTorsoData);
                currentTorsoData(nanMask) = 0;
                torsoData(:,:,frameIdx) = currentTorsoData;
            else
                error('NaN frame found.')
            end
        end
        
        [~, ~, rotationMatrix] = rigidBodyTransformation(EEG.etc.audiomaze.neuralNetworks.referenceTorso, torsoData(:,:,frameIdx));
        identicallyRotatedData = rotationMatrix*torsoData(:,:,frameIdx);
        transitionParameter    = mean(EEG.etc.audiomaze.neuralNetworks.referenceTorso,2) - mean(identicallyRotatedData,2);
        wholeBodyRotatedNotTransitioned = rotationMatrix*phaseSpaceData(1:3,:,frameIdx);
        wholeBodyRotatedAndTransitioned = bsxfun(@plus, wholeBodyRotatedNotTransitioned, transitionParameter);
        wholeBodyRotated(:,:,  frameIdx) = wholeBodyRotatedAndTransitioned;
        rotationMatrixList(:,:,frameIdx) = rotationMatrix;
    end
    
    % vis_mocapMovie(wholeBodyRotated, 500)

    %%%%%%%%%%%%%%%%%%%%
    %%% Fix the arm. %%%
    %%%%%%%%%%%%%%%%%%%%
        
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %%% Interpolate short interruptions (within 1 s). %%%
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        armShortInterruptionFixed  = customInterpolationShortWindow(wholeBodyRotated(:,[12:15],:), 480, 5, 480); % Mocap data sampling rate = 480 Hz.
        handCentroid               = squeeze(mean(wholeBodyRotated(:, handIdx, :), 2));
        armHandCombined            = armShortInterruptionFixed;
        armHandCombined(:,end+1,:) = handCentroid;
        
        
        
        
        
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %%% Recover missing markers. %%%
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        missingMarkerMask = isnan(squeeze(sum(armShortInterruptionFixed,1)));
%         figure; imagesc(missingMarkerMask)

        % Give up frames with no Hand rigid body present.
        handNanIdx = isnan(squeeze(sum(mean(wholeBodyRotated(:,handIdx,:),2))))';
        missingMarkerMask(:,handNanIdx) = false;

        % Missing Ch12.
        fixingIdx = find(missingMarkerMask(1,:) == 1 & sum(missingMarkerMask(2:4,:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_12ch(input);
            armHandCombined(:, 1, fixingIdx) = recovered;
        end
        
        % Missing Ch13.
        fixingIdx = find(missingMarkerMask(2,:) == 1 & sum(missingMarkerMask([1 3 4],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_13ch(input);
            armHandCombined(:, 2, fixingIdx) = recovered;
        end
        
        % Missing Ch14.
        fixingIdx = find(missingMarkerMask(3,:) == 1 & sum(missingMarkerMask([1 2 4],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);       
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_14ch(input);
            armHandCombined(:, 3, fixingIdx) = recovered;
        end
        
        % Missing Ch15.
        fixingIdx = find(missingMarkerMask(4,:) == 1 & sum(missingMarkerMask(1:3,:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_15ch(input);
            armHandCombined(:, 4, fixingIdx) = recovered;
        end

    
        
        
        
        % Missing Ch12-13.
        fixingIdx = find(sum(missingMarkerMask([1 2],:)) == 2 & sum(missingMarkerMask([3 4],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_12_13ch(input);
            recovered3D = reshape(recovered, [3 2 size(recovered,2)]);
            armHandCombined(:, [1 2], fixingIdx) = recovered3D;
        end
        
        % Missing Ch12-14.
        fixingIdx = find(sum(missingMarkerMask([1 3],:)) == 2 & sum(missingMarkerMask([2 4],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_12_14ch(input);
            recovered3D = reshape(recovered, [3 2 size(recovered,2)]);
            armHandCombined(:, [1 3], fixingIdx) = recovered3D;
        end
        
        % Missing Ch12-15.
        fixingIdx = find(sum(missingMarkerMask([1 4],:)) == 2 & sum(missingMarkerMask([2 3],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_12_15ch(input);
            recovered3D = reshape(recovered, [3 2 size(recovered,2)]);
            armHandCombined(:, [1 4], fixingIdx) = recovered3D;
        end
        
        % Missing Ch13-14.
        fixingIdx = find(sum(missingMarkerMask([2 3],:)) == 2 & sum(missingMarkerMask([1 4],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_13_14ch(input);
            recovered3D = reshape(recovered, [3 2 size(recovered,2)]);
            armHandCombined(:, [2 3], fixingIdx) = recovered3D;
        end
        
        % Missing Ch13-15.
        fixingIdx = find(sum(missingMarkerMask([2 4],:)) == 2 & sum(missingMarkerMask([1 3],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_13_15ch(input);
            recovered3D = reshape(recovered, [3 2 size(recovered,2)]);
            armHandCombined(:, [2 4], fixingIdx) = recovered3D;
        end
        
        % Missing Ch14-15.
        fixingIdx = find(sum(missingMarkerMask([3 4],:)) == 2 & sum(missingMarkerMask([1 2],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_14_15ch(input);
            recovered3D = reshape(recovered, [3 2 size(recovered,2)]);
            armHandCombined(:, [3 4], fixingIdx) = recovered3D;
        end
        
        
        
        
        % Missing Ch12-13-14.
        fixingIdx = find(sum(missingMarkerMask([1 2 3],:)) == 3 & sum(missingMarkerMask([4],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_12_13_14ch(input);
            recovered3D = reshape(recovered, [3 3 size(recovered,2)]);
            armHandCombined(:, [1 2 3], fixingIdx) = recovered3D;
        end
        
        % Missing Ch12-13-15.
        fixingIdx = find(sum(missingMarkerMask([1 2 4],:)) == 3 & sum(missingMarkerMask([3],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_12_13_15ch(input);
            recovered3D = reshape(recovered, [3 3 size(recovered,2)]);
            armHandCombined(:, [1 2 4], fixingIdx) = recovered3D;
        end       
        
        % Missing Ch12-14-15.
        fixingIdx = find(sum(missingMarkerMask([1 3 4],:)) == 3 & sum(missingMarkerMask([2],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_12_14_15ch(input);
            recovered3D = reshape(recovered, [3 3 size(recovered,2)]);
            armHandCombined(:, [1 3 4], fixingIdx) = recovered3D;
        end            
        
        % Missing Ch13-14-15.
        fixingIdx = find(sum(missingMarkerMask([2 3 4],:)) == 3 & sum(missingMarkerMask([1],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_13_14_15ch(input);
            recovered3D = reshape(recovered, [3 3 size(recovered,2)]);
            armHandCombined(:, [2 3 4], fixingIdx) = recovered3D;
        end            
        
        
        
        
        
        % Missing Ch12-13-14-15.
        fixingIdx = find(sum(missingMarkerMask([1 2 3 4],:)) == 4);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_12_13_14_15ch(input);
            recovered3D = reshape(recovered, [3 4 size(recovered,2)]);
            armHandCombined(:, [1 2 3 4], fixingIdx) = recovered3D;
        end           
        
        %{
        
        % Validate the results so far.
        figure
        updatadMissingMarkerMask = isnan(squeeze(sum(armHandCombined,1)));
        imagesc(updatadMissingMarkerMask) % Should be empty.
        
        tmp = wholeBodyRotated;
        tmp(:,armIdx,:) = armHandCombined(:,1:4,:);
        vis_mocapMovie(tmp,48)
        
        find(isnan(vec(armHandCombined))) % Should be empty.
        
        %}
        
        
        
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %%% Replace bad data points. %%%
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %%% Evaluate arm marker intervals. %%%
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        armLengthMatrix = [squeeze(sqrt(sum((wholeBodyRotated(:,8,:) -armHandCombined(:,1,:)).^2)))';
                           squeeze(sqrt(sum((armHandCombined(:,1,:)-armHandCombined(:,2,:)).^2)))';
                           squeeze(sqrt(sum((armHandCombined(:,2,:)-armHandCombined(:,3,:)).^2)))';
                           squeeze(sqrt(sum((armHandCombined(:,3,:)-armHandCombined(:,4,:)).^2)))';
                           squeeze(sqrt(sum((armHandCombined(:,4,:)-mean(wholeBodyRotated(:,16:20,:),2)).^2)))'];
                       
        armChannelMatrix = [armLengthMatrix(1,:);...
                            sum(armLengthMatrix([2 3],:));...
                            sum(armLengthMatrix([3 4],:));...
                            armLengthMatrix([5],:)];

        armSumVector = squeeze(sum(sum(armHandCombined,1),2))';
        nonNanIdx    = find(~isnan(armSumVector));

        % Find unbiased mean and SD.
        goodArmChannelMask = zeros(size(armChannelMatrix));
        for channelIdx = 1:size(goodArmChannelMask,1)
            nonNanLengthVector  = armLengthMatrix(channelIdx,:);
            goodLengthThreshold = prctile(nonNanLengthVector, 95);
            goodLengthIdx       = find(nonNanLengthVector<goodLengthThreshold);
            goodLengthData      = nonNanLengthVector(goodLengthIdx);
            goodLengthMean      = mean(goodLengthData);
            goodLengthStd       = std(goodLengthData);
            nonNanGoodIdxIdx    = find(nonNanLengthVector<(goodLengthMean+sdThreshold*goodLengthStd));
            goodArmChannelMask(channelIdx, nonNanGoodIdxIdx) = 1;
        end
        
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %%% Another level of cleaning. %%%
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        goodArmXyz = armHandCombined(:,1:4,nonNanIdx);
        zscoreMask = zeros(size(goodArmXyz,2), size(goodArmXyz,3));
        for chIdx = 1:size(goodArmXyz,2)
            zscoredXyz          = zscore(squeeze(goodArmXyz(:,chIdx,:)), [], 2);
            goodZscoreMask      = zscoredXyz>-4 & zscoredXyz<4;
            goodZscoreMask      = sum(goodZscoreMask)==3;
            zscoreMask(chIdx,:) = goodZscoreMask;
        end
        additionalMask = zeros(size(goodArmChannelMask));
        additionalMask(:,nonNanIdx) = zscoreMask;
        combinedMask = goodArmChannelMask.*additionalMask;
        
        

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %%% Repleace outliers with NaN. %%%
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        missingMarkerMask = ~(logical(combinedMask));
        
        % Give up frames with no Hand rigid body present.
        handNanIdx = isnan(squeeze(sum(mean(wholeBodyRotated(:,16:20,:),2))))';
        missingMarkerMask(:,handNanIdx) = false;
        missingMarkerMask_toApply = repmat(missingMarkerMask, [1 1 3]);
        missingMarkerMask_toApply = permute(missingMarkerMask_toApply, [3 1 2]);
        armHandCombined_armOnly = armHandCombined(:,1:4,:);
        armHandCombined_armOnly(missingMarkerMask_toApply) = NaN;
        armHandCombined(:,1:4,:) = armHandCombined_armOnly;
        
        
        
        % Missing Ch12.
        fixingIdx = find(missingMarkerMask(1,:) == 1 & sum(missingMarkerMask([2 3 4],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_12ch(input);
            armHandCombined(:, 1, fixingIdx) = recovered;
        end
        
        % Missing Ch13.
        fixingIdx = find(missingMarkerMask(2,:) == 1 & sum(missingMarkerMask([1 3 4],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_13ch(input);
            armHandCombined(:, 2, fixingIdx) = recovered;
        end
        
        % Missing Ch14.
        fixingIdx = find(missingMarkerMask(3,:) == 1 & sum(missingMarkerMask([1 2 4],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_14ch(input);
            armHandCombined(:, 3, fixingIdx) = recovered;
        end
        
        % Missing Ch15.
        fixingIdx = find(missingMarkerMask(4,:) == 1 & sum(missingMarkerMask(1:3,:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx); 
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_15ch(input);
            armHandCombined(:, 4, fixingIdx) = recovered;
        end


        
        
        
        % Missing Ch12-13.
        fixingIdx = find(sum(missingMarkerMask([1 2],:)) == 2 & sum(missingMarkerMask([3 4],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx); 
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_12_13ch(input);
            recovered3D = reshape(recovered, [3 2 size(recovered,2)]);
            armHandCombined(:, [1 2], fixingIdx) = recovered3D;
        end
        
        % Missing Ch12-14.
        fixingIdx = find(sum(missingMarkerMask([1 3],:)) == 2 & sum(missingMarkerMask([2 4],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_12_14ch(input);
            recovered3D = reshape(recovered, [3 2 size(recovered,2)]);
            armHandCombined(:, [1 3], fixingIdx) = recovered3D;
        end
        
        % Missing Ch12-15.
        fixingIdx = find(sum(missingMarkerMask([1 4],:)) == 2 & sum(missingMarkerMask([2 3],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_12_15ch(input);
            recovered3D = reshape(recovered, [3 2 size(recovered,2)]);
            armHandCombined(:, [1 4], fixingIdx) = recovered3D;
        end
        
        % Missing Ch13-14.
        fixingIdx = find(sum(missingMarkerMask([2 3],:)) == 2 & sum(missingMarkerMask([1 4],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_13_14ch(input);
            recovered3D = reshape(recovered, [3 2 size(recovered,2)]);
            armHandCombined(:, [2 3], fixingIdx) = recovered3D;
        end
        
        % Missing Ch13-15.
        fixingIdx = find(sum(missingMarkerMask([2 4],:)) == 2 & sum(missingMarkerMask([1 3],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_13_15ch(input);
            recovered3D = reshape(recovered, [3 2 size(recovered,2)]);
            armHandCombined(:, [2 4], fixingIdx) = recovered3D;
        end
        
        % Missing Ch14-15.
        fixingIdx = find(sum(missingMarkerMask([3 4],:)) == 2 & sum(missingMarkerMask([1 2],:)) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_14_15ch(input);
            recovered3D = reshape(recovered, [3 2 size(recovered,2)]);
            armHandCombined(:, [3 4], fixingIdx) = recovered3D;
        end
        

        
        
        
        % Missing Ch12-13-14.
        fixingIdx = find(sum(missingMarkerMask([1 2 3],:)) == 3 & missingMarkerMask([4],:) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_12_13_14ch(input);
            recovered3D = reshape(recovered, [3 3 size(recovered,2)]);
            armHandCombined(:, [1 2 3], fixingIdx) = recovered3D;
        end
        
        % Missing Ch12-13-15.
        fixingIdx = find(sum(missingMarkerMask([1 2 4],:)) == 3 & missingMarkerMask([3],:) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_12_13_15ch(input);
            recovered3D = reshape(recovered, [3 3 size(recovered,2)]);
            armHandCombined(:, [1 2 4], fixingIdx) = recovered3D;
        end       
        
        % Missing Ch12-14-15.
        fixingIdx = find(sum(missingMarkerMask([1 3 4],:)) == 3 & missingMarkerMask([2],:) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_12_14_15ch(input);
            recovered3D = reshape(recovered, [3 3 size(recovered,2)]);
            armHandCombined(:, [1 3 4], fixingIdx) = recovered3D;
        end            
        
        % Missing Ch13-14-15.
        fixingIdx = find(sum(missingMarkerMask([2 3 4],:)) == 3 & missingMarkerMask([1],:) == 0);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_13_14_15ch(input);
            recovered3D = reshape(recovered, [3 3 size(recovered,2)]);
            armHandCombined(:, [2 3 4], fixingIdx) = recovered3D;
        end            
        
        
        
        
        
        % Missing Ch12-13-14-15.
        fixingIdx = find(sum(missingMarkerMask([1 2 3 4],:)) == 4);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            nonNanChIdx = find(~isnan(armHandCombined(1,:,fixingIdx(1))));
            input       = reshape(armHandCombined(:, nonNanChIdx, fixingIdx), [size(armHandCombined,1)*length(nonNanChIdx), length(fixingIdx)]);
            recovered   = EEG.etc.audiomaze.neuralNetworks.net_12_13_14_15ch(input);
            recovered3D = reshape(recovered, [3 4 size(recovered,2)]);
            armHandCombined(:, [1 2 3 4], fixingIdx) = recovered3D;
        end
        
        
        
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %%% Put back the fixed markers. %%%
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        wholeBodyRotated(:,armIdx,:) = armHandCombined(:,1:4,:);
        

        %{
        
        % Validate the results so far.
        figure
        updatadMissingMarkerMask = isnan(squeeze(sum(armHandCombined,1)));
        imagesc(updatadMissingMarkerMask) % Should be empty.
        
        tmp = wholeBodyRotated;
        tmp(:,armIdx,:) = armHandCombined(:,1:4,:);
        vis_mocapMovie(tmp,48)
        
        find(isnan(vec(armHandCombined))) % Should be empty.

        %}
 
        
        
        
    %%%%%%%%%%%%%%%%%%%%%
    %%% Fix the knee. %%%
    %%%%%%%%%%%%%%%%%%%%%
        
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %%% Interpolate short interruptions (within 1 s). %%%
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        rightKneeShortInterruptionFixed = customInterpolationShortWindow(wholeBodyRotated(:,22,:), 480, 5, 480); % Mocap data sampling rate = 480 Hz.
        leftKneeShortInterruptionFixed  = customInterpolationShortWindow(wholeBodyRotated(:,28,:), 480, 5, 480); % Mocap data sampling rate = 480 Hz.
        footR_Centroid                   = mean(wholeBodyRotated(:, footR_Idx, :), 2);
        footL_Centroid                   = mean(wholeBodyRotated(:, footL_Idx, :), 2);
        
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %%% Recover missing markers. %%%
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        
        % Missing Ch22.
        if isfield(EEG.etc.audiomaze.neuralNetworks, 'net_22ch')
            
            rightKneeMissingMarkerMask = isnan(squeeze(sum(rightKneeShortInterruptionFixed,1)));
            fixingIdx = find(rightKneeMissingMarkerMask == 1);
            fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
            if any(fixingIdx)
                input     = cat(2, footR_Centroid, footL_Centroid);
                input     = reshape(input(:,:,fixingIdx), [size(input,1)*size(input,2) length(fixingIdx)]);
                recovered = EEG.etc.audiomaze.neuralNetworks.net_22ch(input);
                wholeBodyRotated(:,22,fixingIdx) = recovered;
            end
            
        else % For subj 829. Simulate all the right knee positions. Finally done (02/17/2020 Makoto).
            
            % Determine rotation matrix to obtain parallel torso.
            referenceTorso = EEG.etc.audiomaze.neuralNetworks.referenceTorso;
            
                    % % Find optimum theta that maximizes x-axis distance between right and left foot centroid.
                    % referenceTorsoCentroid = mean(referenceTorso,2);
                    % 
                    % % Test 1: Use the torso centroid.
                    % referenceTorsoDemeaned = bsxfun(@minus, referenceTorso, mean(referenceTorso,2));
                    % parallelBody_atOrigin  = bsxfun(@minus, wholeBodyRotated, referenceTorsoCentroid); % Depth, Height, Width.
                    % 
                    % % Test 2: Use the half point of waist line.
                    % referenceTorsoDemeaned  = bsxfun(@minus, referenceTorso, mean(referenceTorso(:, [5 6]),2));
                    % parallelBody_atOrigin   = bsxfun(@minus, wholeBodyRotated, mean(referenceTorso(:, [5 6]),2)); % Depth, Height, Width.
                    % 
                    % footCentroidDistInX = [];
                    % for simIdx = 1:101
                    % 
                    %     theta = 0.09+0.0001*simIdx % This makes the x coordinate of footR - footL maximum.
                    %     rotationMatrix = [cos(theta) 0 sin(theta); ...
                    %                      0 1 0; ...
                    %                     -sin(theta) 0 cos(theta)];
                    %     %rotationMatrix = eye(3);
                    % 
                    % 
                    %     % rotationMatrix = eye(3);
                    %     rotatedTorso = rotationMatrix*referenceTorsoDemeaned;
                    %     waistLine2   = rotatedTorso(:, [5 6]);
                    % 
                    %     % Apply the rotation matrix uniformly.
                    %     parallelBody = rotationMatrix*parallelBody_atOrigin(:,:);
                    %     parallelBody = reshape(parallelBody, size(wholeBodyRotated));
                    % 
                    %     parallelBody_footRCent = squeeze(nanmean(parallelBody(:,footR_Idx,:),2));
                    %     parallelBody_footLCent = squeeze(nanmean(parallelBody(:,footL_Idx,:),2));
                    %     footCentroidDistInX(simIdx) = nanmean(parallelBody_footRCent(3,:) - parallelBody_footLCent(3,:));
                    %     [a,b] = max(footCentroidDistInX); % 0.0981.
                    % end
                        
            % Rotate the demeaned reference rigid body.               
            theta = 0.0981; % or pi/32, this makes the x coordinate of footR - footL maximum.
                %theta = 0; % Default angle at the starting position
            rotationMatrix = [cos(theta) 0 sin(theta); 0 1 0; -sin(theta) 0 cos(theta)];
                % rotationMatrix = eye(3);
            refTorsoCentroid = mean(referenceTorso,2);
            parallelBody     = rotationMatrix*wholeBodyRotated(:,:);
            parallelBody     = reshape(parallelBody, size(wholeBodyRotated));

            % Flip LR by multiplying -1.
            parallelBody_LRflipped = bsxfun(@times, parallelBody, [1;1;-1]);
            
            % Rotate it back.
            wholeBodyRotated_LRflipped = inv(rotationMatrix)*parallelBody_LRflipped(:,:);
            wholeBodyRotated_LRflipped = reshape(wholeBodyRotated_LRflipped, size(parallelBody_LRflipped));

%             figure
%             plot(squeeze(mean(wholeBodyRotated(:,torsoIdx,:),2))')
%             hold on
%             plot(squeeze(mean(wholeBodyRotated_LRflipped(:,torsoIdx,:),2))')
            
            % Make sure the torso centroids are the same.
            flippedTorsoCentroid = nanmean(nanmean(wholeBodyRotated_LRflipped(:,torsoIdx,:),2),3);
            wholeBodyRotated_LRflipped_demeaned             = bsxfun(@minus, wholeBodyRotated_LRflipped, flippedTorsoCentroid);
            wholeBodyRotated_LRflipped_torsoCentroidAjusted = bsxfun(@plus, wholeBodyRotated_LRflipped_demeaned, refTorsoCentroid);
            
%             figure
%             plot(squeeze(mean(wholeBodyRotated(:,torsoIdx,:),2))')
%             hold on
%             plot(squeeze(mean(wholeBodyRotated_LRflipped_torsoCentroidAjusted(:,torsoIdx,:),2))')

            % Feed the mirror-flipped footL and footR to net28.
            footR_centroid = squeeze(nanmean(wholeBodyRotated_LRflipped_torsoCentroidAjusted(:,footR_Idx,:),2));
            footL_centroid = squeeze(nanmean(wholeBodyRotated_LRflipped_torsoCentroidAjusted(:,footL_Idx,:),2));
            
                %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
                %%% Another level of cleaning. %%%
                %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
                % To recover the right knee, use 1-Hz low-pass filtered footR and footL centroid data.
                nonnanIdx      = find(~isnan(footR_centroid(1,:)));
                zscoredXyz     = zscore(footR_centroid(:,nonnanIdx), [], 2);
                goodZscoreMask = zscoredXyz>-3 & zscoredXyz<3;
                footR_CorrectionIdx = find(sum(goodZscoreMask)~=3);
                footR_centroid_interpolated = footR_centroid;
                footR_centroid_interpolated(:, nonnanIdx(footR_CorrectionIdx)) = NaN;
                footR_centroid = customInterpolation(footR_centroid_interpolated, 1);
                
                nonnanIdx      = find(~isnan(footL_centroid(1,:)));
                zscoredXyz     = zscore(footL_centroid(:,nonnanIdx), [], 2);
                goodZscoreMask = zscoredXyz>-3 & zscoredXyz<3;
                footL_CorrectionIdx = find(sum(goodZscoreMask)~=3);
                footL_centroid_interpolated = footL_centroid;
                footL_centroid_interpolated(:, nonnanIdx(footL_CorrectionIdx)) = NaN;
                footL_centroid = customInterpolation(footL_centroid_interpolated, 1);               
                
            input          = [footR_centroid; footL_centroid];
            recovered      = EEG.etc.audiomaze.neuralNetworks.net_28ch(input);
            
            %{
            
            tmp = wholeBodyRotated_LRflipped_torsoCentroidAjusted;
            tmp(:,22,:) = recovered;
            vis_mocapMovie(tmp, 48)
            
            %}
            
            % Put the right knee back.
            wholeBodyRotated_LRflipped_torsoCentroidAjusted(:,22,:) = recovered;
            
            % Flip the whole body.
            correctedData = [1 0 0; 0 1 0; 0 0 -1]*wholeBodyRotated_LRflipped_torsoCentroidAjusted(:,:);
            correctedData = reshape(correctedData, size(wholeBodyRotated_LRflipped_torsoCentroidAjusted));
            
            % Rotate it back.
            [fitResult, rmsError, rotationMatrix, centroidOffset] = rigidBodyTransformation(double(referenceTorso), double(nanmean(correctedData(:,torsoIdx,:),3)));
            recoveredWholeBody = rotationMatrix*correctedData(:,:);
            recoveredWholeBody = reshape(recoveredWholeBody, size(correctedData));
            recoveredWholeBody = bsxfun(@minus, recoveredWholeBody, nanmean(nanmean(recoveredWholeBody(:,torsoIdx,:),2),3));
            recoveredWholeBody = bsxfun(@plus, recoveredWholeBody, mean(refTorsoCentroid,2));
            
            %{
            
            % Check.
            tmp = wholeBodyRotated;
            tmp(:,22,:) = recoveredWholeBody(:,22,:);
            vis_mocapMovie(tmp, 48)
            
            %}
            
            wholeBodyRotated(:,22,:) = recoveredWholeBody(:,22,:);
        end
         
        
        % Missing Ch28.
        leftKneeMissingMarkerMask = isnan(squeeze(sum(leftKneeShortInterruptionFixed,1)));
        fixingIdx = find(leftKneeMissingMarkerMask == 1);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            input     = cat(2, footR_Centroid, footL_Centroid);
            input     = reshape(input(:,:,fixingIdx), [size(input,1)*size(input,2) length(fixingIdx)]);
            recovered = EEG.etc.audiomaze.neuralNetworks.net_28ch(input);
            wholeBodyRotated(:,22,fixingIdx) = recovered;
        end
        
        %{
        
        % Validate the results so far.
        figure
        updatadMissingMarkerMask = isnan(squeeze(sum(armHandCombined,1)));
        imagesc(updatadMissingMarkerMask) % Should be empty.
        
        tmp = wholeBodyRotated;
        tmp(:,22,:) = rightKneeShortInterruptionFixed;
        tmp(:,28,:) = leftKneeShortInterruptionFixed;
        vis_mocapMovie(tmp,48)
        
        find(isnan(vec(armHandCombined))) % Should be empty.

        %}
        
        
        

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %%% Replace bad data points. %%%
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        
        % Evaluate knee marker intervals.
        rightKneeLengthMatrix = [squeeze(sqrt(sum((wholeBodyRotated(:,21,:)-rightKneeShortInterruptionFixed).^2)))';
                                 squeeze(sqrt(sum((mean(wholeBodyRotated(:,23:26,:),2)-rightKneeShortInterruptionFixed).^2)))'];   
        leftKneeLengthMatrix  = [squeeze(sqrt(sum((wholeBodyRotated(:,27,:)-leftKneeShortInterruptionFixed).^2)))';
                                 squeeze(sqrt(sum((mean(wholeBodyRotated(:,29:32,:),2)-leftKneeShortInterruptionFixed).^2)))'];        
        rightLegLength = sum(rightKneeLengthMatrix);                    
        leftLegLength  = sum(leftKneeLengthMatrix);                    
        rightLegNonNanIdx = find(~isnan(rightLegLength));
        leftLegNonNanIdx  = find(~isnan(leftLegLength));
        
        % Find unbiased mean and SD for the right leg.
        goodRightLegLengthMask = zeros(size(rightLegLength));
        nonNanLengthVector  = rightLegLength(rightLegNonNanIdx);
        goodLengthThreshold = prctile(nonNanLengthVector, 95);
        goodLengthIdx       = find(nonNanLengthVector<goodLengthThreshold);
        goodLengthData      = nonNanLengthVector(goodLengthIdx);
        goodLengthMean      = mean(goodLengthData);
        goodLengthStd       = std(goodLengthData);
        nonNanGoodIdxIdx    = find(nonNanLengthVector<(goodLengthMean+sdThreshold*goodLengthStd));
        goodRightLegLengthMask(rightLegNonNanIdx(nonNanGoodIdxIdx)) = 1;
        goodRightLegLengthIdx = find(goodRightLegLengthMask);
        
        % Find unbiased mean and SD for the left leg.
        goodLeftLegLengthMask = zeros(size(leftLegLength));
        nonNanLengthVector  = leftLegLength(leftLegNonNanIdx);
        goodLengthThreshold = prctile(nonNanLengthVector, 95);
        goodLengthIdx       = find(nonNanLengthVector<goodLengthThreshold);
        goodLengthData      = nonNanLengthVector(goodLengthIdx);
        goodLengthMean      = mean(goodLengthData);
        goodLengthStd       = std(goodLengthData);
        nonNanGoodIdxIdx    = find(nonNanLengthVector<(goodLengthMean+sdThreshold*goodLengthStd));
        goodLeftLegLengthMask(leftLegNonNanIdx(nonNanGoodIdxIdx)) = 1;  
        goodLeftLegLengthIdx = find(goodLeftLegLengthMask);

        
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %%% Another level of cleaning. %%%
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        goodKneeXyz = rightKneeShortInterruptionFixed(:,:,goodRightLegLengthIdx);
        zscoreMask = zeros(size(goodKneeXyz,2), size(goodKneeXyz,3));
        for chIdx = 1:size(goodKneeXyz,2)
            zscoredXyz          = zscore(squeeze(goodKneeXyz(:,chIdx,:)), [], 2);
            goodZscoreMask      = zscoredXyz>-4 & zscoredXyz<4;
            goodZscoreMask      = sum(goodZscoreMask)==3;
            zscoreMask(chIdx,:) = goodZscoreMask;
        end
        goodXyzIdx = find(zscoreMask==1);
        goodRightLegLengthIdx = goodRightLegLengthIdx(goodXyzIdx);
        
        goodKneeXyz = leftKneeShortInterruptionFixed(:,:,goodLeftLegLengthIdx);
        zscoreMask = zeros(size(goodKneeXyz,2), size(goodKneeXyz,3));
        for chIdx = 1:size(goodKneeXyz,2)
            zscoredXyz          = zscore(squeeze(goodKneeXyz(:,chIdx,:)), [], 2);
            goodZscoreMask      = zscoredXyz>-4 & zscoredXyz<4;
            goodZscoreMask      = sum(goodZscoreMask)==3;
            zscoreMask(chIdx,:) = goodZscoreMask;
        end
        goodXyzIdx = find(zscoreMask==1);
        goodLeftLegLengthIdx = goodLeftLegLengthIdx(goodXyzIdx);       
        

        
        if isfield(EEG.etc.audiomaze.neuralNetworks, 'net_22ch')
            % Re-do Ch22.
            fixingIdx = setdiff(1:length(rightKneeShortInterruptionFixed), goodRightLegLengthIdx);
            fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
            if any(fixingIdx)
                input     = reshape(cat(2, footR_Centroid(:,:,fixingIdx), footL_Centroid(:,:,fixingIdx)), [3*2 length(fixingIdx)]);
                recovered = EEG.etc.audiomaze.neuralNetworks.net_22ch(input);
                wholeBodyRotated(:,22,fixingIdx) = recovered;
            end
        end
        
        % Re-do Ch28.
        fixingIdx = setdiff(1:length(leftKneeShortInterruptionFixed), goodLeftLegLengthIdx);
        fixingIdx = intersect(fixingIdx, torsoNonNanFrameIdx);
        if any(fixingIdx)
            input     = reshape(cat(2, footR_Centroid(:,:,fixingIdx), footL_Centroid(:,:,fixingIdx)), [3*2 length(fixingIdx)]);
            recovered = EEG.etc.audiomaze.neuralNetworks.net_28ch(input);
            wholeBodyRotated(:,28,fixingIdx) = recovered;
        end


        %{
        
        % Final visual confirmation.
        vis_mocapMovie(wholeBodyRotated, 48)
        
        find(isnan(vec(wholeBodyRotated(:,headIdx,:))))
        find(isnan(vec(wholeBodyRotated(:,torsoIdx,:))))
        find(isnan(vec(wholeBodyRotated(:,handIdx,:))))
        find(isnan(vec(wholeBodyRotated(:,footR_Idx,:))))
        find(isnan(vec(wholeBodyRotated(:,footL_Idx,:))))
        find(isnan(vec(wholeBodyRotated(:,armIdx,:))))
        find(isnan(vec(wholeBodyRotated(:,kneeIdx,:))))
        
        %}
        
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %%% Rotate them back to the original position. %%%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    for goodFrameIdxIdx = 1:length(torsoNonNanFrameIdx)
        if mod(goodFrameIdxIdx, 10000) == 0
            disp(sprintf('%.0f/%.0f...', goodFrameIdxIdx, length(torsoNonNanFrameIdx)));
        end
        frameIdx = torsoNonNanFrameIdx(goodFrameIdxIdx);
        
        % Fix NaN for one of the xyz coordinates that happened to be genuinly zero, 
        currentTorsoData = torsoData(:,:,frameIdx);
        if any(isnan(currentTorsoData(:)))
            xyzNanCheck = find(sum(isnan(currentTorsoData)) == 3);
            if isempty(xyzNanCheck)
                nanMask = isnan(currentTorsoData);
                currentTorsoData(nanMask) = 0;
                torsoData(:,:,frameIdx) = currentTorsoData;
            else
                error('NaN frame found.')
            end
        end
        
        currentBackRotationMatrix = inv(rotationMatrixList(:,:,frameIdx));
        identicallyRotatedData = currentBackRotationMatrix*wholeBodyRotated(:,:,frameIdx);
        transitionParameter    = mean(torsoData(:,:,frameIdx),2) - mean(identicallyRotatedData(:,torsoIdx),2);
        wholeBodyRotatedAndTransitioned = bsxfun(@plus, identicallyRotatedData, transitionParameter);
        phaseSpaceData(1:3,:,frameIdx) = wholeBodyRotatedAndTransitioned;
    end
    
    % Store the results.
    EEG.etc.audiomaze.phaseSpaceCorrected = phaseSpaceData;
    
    % Save the data.
    pop_saveset(EEG, 'filename', EEG.filename, 'filepath', '/data/mobi/Audiomaze/makoto/p06_applyNeuralNetwork')
end