"""
BMP image test for Adafruit MagTag. Demonstrates some loading and/or
rendering issues and differences in displayio vs imagereader library.

Requires these libs (not all used in this code, but referenced by magtag):
adafruit_bitmap_font
adafruit_display_text
adafruit_il0373
adafruit_imageload
adafruit_io
adafruit_magtag
adafruit_requests
neopixel
simpleio

Requires these images in the 'bitmaps' folder:
lowbatt.bmp
textpattern4.bmp
textpattern8.bmp
textpattern8rle.bmp
textpattern24.bmp
"""

import displayio
import adafruit_imageload
from adafruit_magtag.magtag import Graphics

GRAPHICS = Graphics(auto_refresh=False)
DISPLAY = GRAPHICS.display

def fillrect(xpos, ypos, width, height, color):
    palette = displayio.Palette(1)
    palette[0] = color
    bitmap = displayio.Bitmap(width, 1, 1)
    return displayio.TileGrid(bitmap, pixel_shader=palette, x=xpos, y=ypos,
                              width=1, height=height)

def img_disk(filename, xpos, ypos):
    """ Load a BMP image using displayio.OnDiskBitmap """
    file = open(filename, 'rb')
    bitmap = displayio.OnDiskBitmap(file)
    return displayio.TileGrid(bitmap, x=xpos, y=ypos,
                              pixel_shader=displayio.ColorConverter())
    return tilegrid

def img_ram(filename, xpos, ypos):
    """ Load a BMP image using adafruit_imageload """
    image, palette = adafruit_imageload.load(filename,
                                             bitmap=displayio.Bitmap,
                                             palette=displayio.Palette)
    return displayio.TileGrid(image, pixel_shader=palette, x=xpos, y=ypos)


GROUP = displayio.Group(max_size=8)
# Background is black by default. Add some alternating white bars:
GROUP.append(fillrect(0, DISPLAY.height // 8, DISPLAY.width,
                      DISPLAY.height // 4, 0xFFFFFF))
GROUP.append(fillrect(0, DISPLAY.height // 8 + DISPLAY.height // 2,
                      DISPLAY.width, DISPLAY.height // 4, 0xFFFFFF))

# Load some images using OnDiskBitmap (top row of 3 images)...
# This 4-bit BMP causes board to crash hard:
#GROUP.append(img_disk('bitmaps/testpattern4.bmp', 0, 0))
# But this one (also a 4-bit BMP, but different size) loads OK:
GROUP.append(img_disk('bitmaps/lowbatt.bmp', 0, 0))
# Notice however that the black pixels in the image are instead transparent
# (though not requested as such), so the background bars show through.
# Workaround for now is to draw a black rectangle behind the image
# (not done here, but works in other code).
# As for why the first image crashes -- possibly the image dimensions?
# The test patterns are prime on both axes, battery image is not.
# 8-bit BMP loads fine:
GROUP.append(img_disk('bitmaps/testpattern8.bmp', DISPLAY.width // 3, 0))
# As does 24-bit BMP:
GROUP.append(img_disk('bitmaps/testpattern24.bmp', DISPLAY.width * 2 // 3, 0))

# Now load same set of images using imageload library instead (bottom row)...
# 4-bit image (the one that OnDiskBitmap couldn't handle) loads and
# displays (and is opaque) but is thresholded to 1-bit:
GROUP.append(img_ram('bitmaps/testpattern4.bmp', 0, DISPLAY.height // 2))
# Same for 8-bit image, is thresholded to 1-bit:
GROUP.append(img_ram('bitmaps/testpattern8.bmp', DISPLAY.width // 3,
                     DISPLAY.height // 2))
# The prior item can also be successfully switched out for
# 'testpattern8.bmp', a slightly smaller RLE compressed file --
# OnDiskBitmap doesn't handle RLE, imageload does. Cool.
# True color BMP is not supported by imageload -- that's OK, already known:
#GROUP.append(img_ram('bitmaps/testpattern24.bmp', DISPLAY.width * 2 // 3,
#                     DISPLAY.height // 2))

DISPLAY.show(GROUP)
DISPLAY.refresh()

while True:
    pass
