# SPDX-FileCopyrightText: Copyright (c) 2021 Kattni Rembor for Adafruit Industries
#
# SPDX-License-Identifier: Unlicense
"""
GitHub Common Phrases MacroPad HotKey Example.

REQUIRES shortcuts.py FILE TO BE INCLUDED ON CIRCUITPY.

This example utilises common phrases I use on GitHub PRs and issues. To update what phrases are
being sent and the associated labels displayed, modify the shortcuts.py file.

This example expects the MacroPad to be rotated 90 degrees, (so that the USB cable is pointing to
the left). It displays an initialism of the phrase on the display, and when the associated key is
pressed, sends the full phrase string via HID. The key labels are arranged on the display in three
groups of four, in a 2x2 layout where each group of four labels represents a row of keys, in which
the top two labels in each group are for the first two keys in the row, and the bottom two labels
are for the last two keys in the row. The groups are delineated by a white line on the display. It
lights up each key that is pressed a color of the rainbow using the NeoPixel LEDs.

This example includes an option for Fidget Mode, which when enabled, disables the HID sending to
allow you to "fidget" with the light-up keys without sending errant strings to your computer. Press
the rotary encoder switch to enable or disable Fidget Mode. It defaults to being enabled when the
code is first run. Once disabled, it will automatically be enabled following a customisable delay.
The little red LED on the back of the board next to the USB connector lights up when Fidget Mode is
enabled.
"""
import time
import displayio
import vectorio
import terminalio
from rainbowio import colorwheel
from adafruit_displayio_layout.layouts.grid_layout import GridLayout
from adafruit_display_text import bitmap_label as label
from adafruit_macropad import MacroPad
from shortcuts import shortcut_keys

# The delay in seconds until Fidget Mode is automatically enabled. This is essentially a timeout on
# Fidget Mode being disabled. This delay begins when Fidget Mode is manually disabled by pressing
# the rotary encoder switch, and, begins again when a key is pressed while Fidget Mode is disabled.
fidget_mode_delay = 15

macropad = MacroPad(rotation=90)

# Set up title.
main_group = displayio.Group()
macropad.display.show(main_group)
title = label.Label(
    y=4,
    font=terminalio.FONT,
    color=0x0,
    text=" SHORTCUTS ",
    background_color=0xFFFFFF,
)
# Set up grid layout.
layout = GridLayout(x=0, y=10, width=64, height=120, grid_size=(2, 6), cell_padding=3,
                    divider_lines=True,
                    # h_divider_line_rows=(2, 4), v_divider_line_cols=(),
                    )

# Extract data from shortcuts file.
label_names = [names[0] for names in shortcut_keys["macros"]]
keys = [keys[2] for keys in shortcut_keys["macros"]]

# Generate the labels based on the label names and add them to the appropriate grid cell.
labels = []
for index in range(12):
    x = index % 2
    y = index // 2
    labels.append(label.Label(terminalio.FONT, text=label_names[index]))
    layout.add_content(labels[index], grid_position=(x, y), cell_size=(1, 1))

# # Create the palette for the divider lines.
# line_palette = displayio.Palette(1)
# line_palette[0] = 0xFFFFFF
# # Generate a 1px tall, 64px wide "rectangle" to display a line between each section of key labels.
# line_one = vectorio.Rectangle(width=64, height=1, x=0, y=51, pixel_shader=line_palette)
# line_two = vectorio.Rectangle(width=64, height=1, x=0, y=91, pixel_shader=line_palette)

# Display the text.
main_group.append(title)
main_group.append(layout)
# # Display the divider lines.
# main_group.append(line_one)
# main_group.append(line_two)

# Begin with Fidget Mode on.
fidget_mode = True
# Get the initial time the first time the code runs.
initial_time = time.monotonic()
while True:
    # Get the now time each time through the loop. Used to track when to enable Fidget Mode.
    now_time = time.monotonic()

    # Begin collecting key events, e.g. key presses.
    key_event = macropad.keys.events.get()
    # Begin collecting rotary encoder switch events, e.g. switch presses.
    macropad.encoder_switch_debounced.update()

    # If there is a key event...
    if key_event:
        # ...and it's that a key has been pressed...
        if key_event.pressed:
            print(key_event.key_number)
            # Highlight the associated label on the display with a white background.
            layout[key_event.key_number].color = 0x000000
            layout[key_event.key_number].background_color = 0xFFFFFF

            # Light up the key pressed a color of the rainbow.
            macropad.pixels[key_event.key_number] = colorwheel(
                int(255 / 12) * key_event.key_number
            )

            # If Fidget Mode is disabled, begin checking on what type of HID to send.
            if not fidget_mode:
                initial_time = time.monotonic()
                # If it's a string...
                if "KL" in shortcut_keys["macros"][key_event.key_number][1]:
                    # ... send the associated string.
                    for key in keys[key_event.key_number]:
                        macropad.keyboard_layout.write(key)

                # Though this example was designed entirely to send strings, I wanted to maintain
                # compatibility with sending other types of HID commands, so I have still included
                # the code to send Keycodes and ConsumerControlCodes.

                # If it's a Keycode...
                if "KC" in shortcut_keys["macros"][key_event.key_number][1]:
                    # ... send the associated key command or sequence of key commands.
                    for key in keys[key_event.key_number]:
                        macropad.keyboard.press(key)
                    macropad.keyboard.release_all()

                # If it's a ConsumerControlCode...
                if "CC" in shortcut_keys["macros"][key_event.key_number][1]:
                    # ... send the associated consumer control code.
                    for key in keys[key_event.key_number]:
                        macropad.consumer_control.send(key)
        else:
            # Otherwise, turn off the LEDs...
            macropad.pixels.fill((0, 0, 0))
            # ...and return the label to white text on a black background.
            layout[key_event.key_number].color = 0xFFFFFF
            layout[key_event.key_number].background_color = 0x000000

    # Disables the HID sending so you can fidget with the keys!
    # Press once to enable Fidget Mode, and once to disable.
    if macropad.encoder_switch_debounced.pressed:
        fidget_mode = not fidget_mode
        # Reset initial_time to continue tracking after switch has been pressed.
        initial_time = time.monotonic()

    # If Fidget Mode is off, and 15 seconds has passed since it was disabled...
    if not fidget_mode and now_time - initial_time > fidget_mode_delay:
        # ...turn on Fidget Mode...
        fidget_mode = True
        # ...and set initial_time equal to now_time to continue time tracking for Fidget Mode.
        initial_time = now_time

    # The little red LED is on when Fidget Mode is enabled.
    macropad.red_led = fidget_mode
